package ru.yandex.tours.wizard.index

import java.io.File
import java.nio.ByteBuffer

import org.apache.commons.io.output.ByteArrayOutputStream
import ru.yandex.tours.index.{IndexWriter, WizardIndexing}
import ru.yandex.tours.index.composite.CompositeIndex
import ru.yandex.tours.index.shard.IndexShard
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.wizard.resource.FreshResourceHolder

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 04.03.15
 */
class FreshIndexHolder(val ttl: FiniteDuration, val name: String)
  extends FreshResourceHolder[IndexShard] with MonitoredIndexHolder {

  protected def buildResource(): IndexShard = FreshIndexHolder.join(shards)

  def size: Int = resource.get.size
}

object FreshIndexHolder extends Logging {

  def fromDirectory(name: String, dir: File, ttl: FiniteDuration): FreshIndexHolder = {
    val holder = new FreshIndexHolder(ttl, name)
    holder.addShards(IndexShard.allFromDir(dir, WizardIndexing.indexExtension))
    holder
  }

  def join(shards: Vector[IndexShard]): IndexShard = {
    log.info(s"Joining ${shards.size} fresh shards")
    val baos = new ByteArrayOutputStream()
    IO.using(new IndexWriter(baos)) { writer =>
      val freshness = shards.foldLeft(System.currentTimeMillis()) { _ min _.freshness }
      writer.writeHeader(freshness, -1)
      CompositeIndex.optimized(shards)
        .iterator
        .foreach(writer.writeItem)
    }
    val buffer = ByteBuffer.wrap(baos.toByteArray)
    IndexShard.fromBuffer(buffer)
  }
}