package ru.yandex.tours.wizard.parser

import ru.yandex.tours.geo.Departures
import ru.yandex.tours.hotels.HotelRatings
import ru.yandex.tours.query._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.wizard.WizardTracer
import ru.yandex.tours.wizard.domain.{TourStartInterval, ToursWizardRequest}
import ru.yandex.tours.wizard.geoaddr.GeoAddrPragmatics
import ru.yandex.tours.wizard.query._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 23.01.15
 */
class WizardRequestParser(reporter: BadQueryReporter,
                          geoAddrPragmatics: GeoAddrPragmatics,
                          userRequestParser: UserRequestParser,
                          departures: Departures,
                          hotelRatings: HotelRatings) extends Logging {

  def parse(request: WizardRequest): Option[ToursWizardRequest] = {
    if (userRequestParser.haveStopWords(request.userRequest)) {
      reporter.stopWords(request.userRequest)
      if (log.isDebugEnabled) {
        log.debug(s"Found stop word in query: [${request.userRequest}]")
      }
      return Option.empty
    }
    WizardTracer.checkpoint("stop_words")

    val geoAddr = request.geoAddr
    WizardTracer.checkpoint("geo_addr")

    val geoParts = geoAddrPragmatics.translate(request.userRequest, geoAddr)
    if (log.isDebugEnabled) {
      log.debug("Geo parts from GeoAddr: " + geoParts.mkString(" "))
    }
    WizardTracer.checkpoint("geo_parts")

    geoParts.foreach(_.boost = 0.9d)
    val parsedQuery = userRequestParser.parse(request.userRequest, geoParts)
    if (parsedQuery.isEmpty) {
      reporter.noMarker(request.userRequest)
      return Option.empty
    }
    WizardTracer.checkpoint("parsed")

    if (log.isDebugEnabled) {
      log.debug(s"Parsed [${parsedQuery.userRequest}]: " + parsedQuery.queryParts.mkString(" "))
    }

    extractToursRequest(parsedQuery, request.userRegion)
  }

  def extractToursRequest(parsedQuery: ParsedUserQuery, userRegion: Int): Option[ToursWizardRequest] = {
    if (parsedQuery.countOf[HotelName] +
      parsedQuery.countOf[HotelNameTop] +
      parsedQuery.countOf[HotelNamePart] +
      parsedQuery.countOf[ArrivalRegion] +
      parsedQuery.countOf[TourMarker.type] +
      parsedQuery.countOf[SkiResorts.type] +
      parsedQuery.countOf[HotelMarker] +
      parsedQuery.countOf[TourOperatorMarker] == 0) {
      return Option.empty
    }

    val hasHotMarker = parsedQuery.exists(_ == HotMarker)

    if (parsedQuery.exists(_ == Unknown)) {
      reporter.unknownParts(parsedQuery)
      return Option.empty
    }

    if (parsedQuery.has[SomeRegion]) {
      reporter.someRegions(parsedQuery)
      return Option.empty
    }

    if (parsedQuery.countOf[DepartureRegion] > 1) {
      reporter.manyDeparture(parsedQuery)
      return Option.empty
    }

    if (parsedQuery.countOf[ArrivalRegion] > 1) {
      reporter.manyArrival(parsedQuery)
      return Option.empty
    }

    def getHotelId(hotelIds: Set[Int]): Option[Int] = {
      if (hotelIds.size > 1) {
        val ratings = hotelRatings.copy()
        Some(hotelIds.maxBy(ratings.getRelevance))
      } else hotelIds.headOption
    }

    val hotel = getHotelId(parsedQuery.collectOf[HotelNameTop].map(_.hotelId).toSet) orElse
      getHotelId(parsedQuery.collectOf[HotelName].map(_.hotelId).toSet)

    if (parsedQuery.has[HotelNamePart]) {
      return Option.empty
    }

    if (parsedQuery.countOf[TourOperatorMarker] > 1) {
      reporter.manyOperators(parsedQuery)
      return Option.empty
    }

    val markers = parsedQuery.collectOf[Marker].toSet

    val from = parsedQuery.collectFirst {
      case DepartureRegion(regionId) => regionId
    }

    val when = parsedQuery.collectFirst {
      case dateInterval: DateInterval => TourStartInterval.fromDateInterval(dateInterval)
    }

    val to = parsedQuery.collectFirst {
      case ArrivalRegion(regionId) => regionId
    }

    val operator = parsedQuery.collectFirst {
      case TourOperatorMarker(operatorId) => operatorId
    }

    val (departure :: fallback) = {
      val fromOrUserRegion = from.getOrElse(userRegion)
      to.fold(departures.getDepartures(fromOrUserRegion)){
        departures.getDepartures(fromOrUserRegion, _)
      }
    }

    Some(ToursWizardRequest(
      parsedQuery.userRequest,
      userRegion = userRegion,
      regionInRequest = from,
      markers = markers,
      from = departure,
      fallbackFrom = fallback,
      when = when.getOrElse(TourStartInterval.default(hasHotMarker)),
      to = to,
      hotel = hotel,
      operator = operator,
      hotelOnTop = parsedQuery.has[HotelNameTop]
    ))
  }
}
