package ru.yandex.tours.wizard.query

import ru.yandex.tours.query.{Pragmatic, Unknown}
import ru.yandex.tours.util.text.StringNormalizer.isSeparator
import ru.yandex.tours.wizard.query.ParsedUserQuery.QueryPart

import scala.reflect._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.01.15
 */
case class ParsedUserQuery(userRequest: String, queryParts: Vector[QueryPart]) {
  import ru.yandex.tours.wizard.query.ParsedUserQuery._

  def isEmpty: Boolean = queryParts.isEmpty
  def nonEmpty: Boolean = queryParts.nonEmpty

  def +(anotherParts: Seq[QueryPart]): ParsedUserQuery =
    copy(queryParts = queryParts ++ anotherParts)

  def map(f: QueryPart => QueryPart): ParsedUserQuery = {
    copy(queryParts = queryParts.map(f))
  }

  def map2(f: ParsedUserQuery => QueryPart => QueryPart): ParsedUserQuery = {
    val f2 = f(this)
    map(f2)
  }

  def flatMap2(f: ParsedUserQuery => QueryPart => Traversable[QueryPart]): ParsedUserQuery = {
    val f2 = f(this)
    copy(queryParts = queryParts.flatMap(f2))
  }

  def filter(f: QueryPart => Boolean): ParsedUserQuery =
    copy(queryParts = queryParts.filter(f))

  def has[T <: Pragmatic: ClassTag]: Boolean =
    queryParts.exists(p => classTag[T].runtimeClass.isInstance(p.pragmatic))

  def countOf[T <: Pragmatic: ClassTag]: Int =
    queryParts.count(p => classTag[T].runtimeClass.isInstance(p.pragmatic))

  def collectOf[T <: Pragmatic: ClassTag]: Seq[T] =
    queryParts.collect {
      case p if classTag[T].runtimeClass.isInstance(p.pragmatic) => p.pragmatic.asInstanceOf[T]
    }

  def exists(f: Pragmatic => Boolean): Boolean =
    queryParts.exists(part => f(part.pragmatic))

  def collect[T](pf: PartialFunction[Pragmatic, T]): Seq[T] =
    queryParts.map(_.pragmatic).collect(pf)

  def collectFirst[T](pf: PartialFunction[Pragmatic, T]): Option[T] =
    queryParts.map(_.pragmatic).collectFirst(pf)

  def selectBestMatch: ParsedUserQuery = {
    if (userRequest.length > MAX_REQUEST_LENGTH) return copy(queryParts = Vector.empty)

    val sorted = queryParts.sortBy(p => (p.startPosition, -p.score)).toList

    var bestScore = 0d
    var bestChain = Vector.empty[ParsedUserQuery.QueryPart]

    val bestScores = Array.fill(userRequest.length + 1)(0d)

    def loop(score: Double, parts: List[QueryPart], chain: Vector[QueryPart], bestScores: Array[Double]): Unit = {
      parts match {
        case Nil =>
          if (score > bestScore) {
            bestScore = score
            bestChain = chain
          }
        case head :: tail =>
          val newEnd = head.endPosition
          val newScore = score + head.score
          if (bestScores(newEnd) < newScore) {
            bestScores(newEnd) = newScore
            loop(newScore, parts.dropWhile(_.startPosition < newEnd), chain :+ head, bestScores)
            if (tail.nonEmpty && tail.head.startPosition < head.endPosition) {
              loop(score, tail, chain, bestScores)
            }
          }
      }
    }
    loop(0d, sorted, Vector.empty, bestScores)
    copy(queryParts = bestChain)
  }

  def containHoles: Boolean = {
    val sorted = queryParts.sortBy(p => (p.startPosition, p.endPosition))
    var prevStart = 0
    var prevEnd = 0

    if (sorted.isEmpty) {
      return true
    }

    val it = sorted.iterator
    while (it.hasNext) {
      val p = it.next()
      if (p.startPosition <= prevEnd) {
        prevStart = p.startPosition
        prevEnd = p.endPosition
      } else {
        return true
      }
    }
    false
  }

  def fillHoles: ParsedUserQuery = {
    val result = Vector.newBuilder[QueryPart]

    var prevEnd = 0

    for (part <- queryParts) {
      if (part.startPosition > prevEnd) {
        result += ParsedUserQuery.QueryPart(userRequest, prevEnd, part.startPosition, Unknown)
      }

      result += part
      prevEnd = part.endPosition
    }

    if (prevEnd < userRequest.length) {
      result += ParsedUserQuery.QueryPart(userRequest, prevEnd, userRequest.length, Unknown)
    }

    copy(queryParts = result.result())
  }
}

object ParsedUserQuery {

  private val MAX_REQUEST_LENGTH = 10000

  def apply(parts: (String, Pragmatic)*): ParsedUserQuery = {
    val userRequest = parts.map(_._1).mkString
    var pos = 0
    val res = Vector.newBuilder[QueryPart]
    for ((slice, pragmatic) <- parts) {
      res += QueryPart(userRequest, pos, pos + slice.length, pragmatic)
      pos += slice.length
    }
    ParsedUserQuery(userRequest, res.result())
  }

  case class QueryPart(userRequest: String, startPosition: Int, endPosition: Int, pragmatic: Pragmatic,
                       var boost: Double = 1.0d) {
    def length: Int = endPosition - startPosition
    def score: Double = (length - 1) * boost

    def atWordStart: Boolean = startPosition == 0 || isSeparator(userRequest.charAt(startPosition - 1))

    def stealPrefix(prefix: String, newPragmatic: Pragmatic = pragmatic): QueryPart = {
      if (userRequest.regionMatches(startPosition, prefix, 0, prefix.length)) {
        if (pragmatic ne newPragmatic) copy(pragmatic = newPragmatic)
        else this
      } else if (startPosition >= prefix.length &&
        userRequest.regionMatches(startPosition - prefix.length, prefix, 0, prefix.length) &&
        (startPosition - prefix.length == 0 || isSeparator(userRequest.charAt(startPosition - prefix.length - 1)))
      ) {
        copy(startPosition = startPosition - prefix.length, pragmatic = newPragmatic)
      } else {
        this
      }
    }

    def slice: String = userRequest.substring(startPosition, endPosition)

    def collide(anotherPart: QueryPart): Boolean = {
      (startPosition == anotherPart.endPosition) || (endPosition == anotherPart.startPosition)
    }

    def isAfter(anotherPart: QueryPart): Boolean = startPosition > anotherPart.startPosition

    def union(anotherPart: QueryPart, pragmatic: Pragmatic): QueryPart = {
      val boost = this.boost min anotherPart.boost
      if (startPosition == anotherPart.endPosition) {
        copy(startPosition = anotherPart.startPosition, pragmatic = pragmatic, boost = boost)
      } else if (endPosition == anotherPart.startPosition) {
        copy(endPosition = anotherPart.endPosition, pragmatic = pragmatic, boost = boost)
      } else sys.error("Parts do not collide")
    }

    override def toString: String = s"$pragmatic${if (boost != 1d) ":" + boost else ""}{$slice}"
  }
}
