package ru.yandex.tours.wizard.search

import java.util.concurrent.atomic.AtomicInteger

import ru.yandex.tours.direction.{Direction, Directions, DirectionsUtils, HotDirections}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.hotels.{HotelRatings, HotelsIndex}
import ru.yandex.tours.index.WizardIndexing
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.query.NoVisa
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.{Top1, Top3}
import ru.yandex.tours.wizard.WizardTracer
import ru.yandex.tours.wizard.domain.Orderings._
import ru.yandex.tours.wizard.domain._
import ru.yandex.tours.wizard.index.IndexHolder

import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.03.15
 */
class IndexToursSearcher(indexHolder: IndexHolder,
                         hotelsIndex: HotelsIndex,
                         hotelRatings: HotelRatings,
                         tree: region.Tree,
                         directions: Directions,
                         hotDirections: HotDirections,
                         allowFallback: Boolean) extends WizardToursSearcher with Logging {

  import IndexToursSearcher._

  private val maxHotelsForDirection = 3
  private val maxDirections = 4

  override def search(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    if (request.hasSkiMarker || Regions.isSkyResort(request.to, directions)) {
      None
    } else if (request.hotel.isDefined) {
      for {
        hotelId <- request.hotel
        hotel <- hotelsIndex.getHotelById(hotelId)
        response <- searchHotel(request, hotel)
      } yield response
    } else if (request.to.isDefined) {
      if (Regions.ignoreSearch(request.to.get)) {
        None
      } else if (!request.hasHotelMarker || Regions.preferTours(request.to.get, tree)) {
        if (request.hasHotMarker) {
          searchHotTours(request)
        } else {
          searchTours(request)
        }
      } else {
        None
      }
    } else if (request.hasHotelMarker) {
      None
    } else {
      searchToursDirections(request)
    }
  }

  private def searchHotel(request: ToursWizardRequest, hotel: Hotel): Option[ToursWizardResponse] = {
    if (request.hasMarker) {
      searchHotelPrices(request, hotel)
    } else {
      val dateWithBestPrice = searchHotelBestPrice(request, hotel)
      if (dateWithBestPrice.isDefined || allowFallback) {
        Some(HotelSnippetResponse(request.from, dateWithBestPrice, hotel))
      } else {
        None
      }
    }
  }

  private def searchHotelBestPrice(request: ToursWizardRequest, hotel: Hotel) = {
    val index = indexHolder.getIndex
    val operatorId = request.operator.getOrElse(0)
    val iterator = index.find(operatorId, request.from, WizardIndexing.toHotel(hotel.id), request.when.from, request.when.until)
    var when = 0
    var minPrice = Int.MaxValue
    var nights = 0
    var i = 0
    for (e <- iterator) {
      i += 1
      if (e.minPrice < minPrice) {
        minPrice = e.minPrice
        when = e.when
        nights = e.nights
      }
    }
    WizardTracer.checkpoint("hotel_best_price_finish", i)
    if (minPrice < Int.MaxValue) Some(HotelPrice(when, nights, minPrice))
    else None
  }

  private def searchHotelPrices(request: ToursWizardRequest, hotel: Hotel): Option[ToursWizardResponse] = {
    val operatorId = request.operator.getOrElse(0)

    val index = indexHolder.getIndex
    val iterator = index.find(operatorId, request.from, WizardIndexing.toHotel(hotel.id), request.when.from, request.when.until)

    val tops = new mutable.HashMap[Int, Top1[HotelPrice]]

    var i = 0
    val start = System.currentTimeMillis()
    for (e <- iterator) {
      i += 1
      val price = HotelPrice(e.when, e.nights, e.minPrice)
      tops.getOrElseUpdate(price.nights, new Top1) += price
    }
    WizardTracer.checkpoint("hotel_prices_finish", i)
    log.debug(s"HOTEL_PRICES: Traversed $i elements in ${System.currentTimeMillis() - start} ms.")

    val withPrice = tops.valuesIterator.filter(_.nonEmpty).map(_.head).toArray
    if (withPrice.length >= maxHotelsForDirection) {
      val prices = withPrice.sortBy { price =>
        math.abs(price.nights - 7) min math.abs(price.nights - 10) min math.abs(price.nights - 14)
      }.take(maxHotelsForDirection).sortBy(_.nights)
      Some(HotelPricesResponse(
        request.from,
        hotel,
        prices
      ))
    } else if (allowFallback) {
      val bestPrice =
        if (withPrice.nonEmpty) Some(withPrice.min)
        else None
      Some(HotelSnippetResponse(
        request.from,
        bestPrice,
        hotel
      ))
    } else {
      None
    }
  }

  private def joinWithHotels(hotelPrices: List[HotelIdWithPrice]) = {
    val foundHotels = hotelsIndex.getHotelsById(hotelPrices.map(_.hotelId))
    for {
      hotelIdWithPrice <- hotelPrices
      hotel <- foundHotels.get(hotelIdWithPrice.hotelId)
    } yield hotelIdWithPrice.toResponse(hotel)
  }

  private def searchHotTours(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    val operatorId = request.operator.getOrElse(0)

    val index = indexHolder.getIndex
    val iterator = index.find(operatorId, request.from, request.to.get, request.when.from, request.when.until)

    val cheapestHotels = new Top3[HotelIdWithPrice, Int](_.hotelId)(HotelIdWithPrice.byPrice, HotelIdWithPrice.byPrice)

    var i = 0
    val start = System.currentTimeMillis()
    for {
      e <- iterator
    } {
      i += 1
      val price = new HotelIdWithPrice(e.when, e.nights, e.hotelId, e.minPrice, Pansion.valueOf(e.pansion))
      cheapestHotels += price
    }
    WizardTracer.checkpoint("hotels_finish", i)
    log.debug(s"HOTELS: Traversed $i elements in ${System.currentTimeMillis() - start} ms.")

    if (cheapestHotels.size >= maxHotelsForDirection) {
      val hotels = cheapestHotels.toList
      val list = joinWithHotels(hotels).take(maxHotelsForDirection)
      if (list.size == maxHotelsForDirection) {
        Some(ToursResponse(
          request.from,
          request.to.get,
          list
        ))
      } else {
        None
      }
    } else {
      None
    }
  }

  private def searchTours(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    val operatorId = request.operator.getOrElse(0)

    val index = indexHolder.getIndex
    val iterator = index.find(operatorId, request.from, request.to.get, request.when.from, request.when.until)

    val cheapestHotel = new Top1[HotelIdWithPrice]()(HotelIdWithPrice.byPrice)
    val topHotels = new Top3[HotelIdWithPrice, Int](_.hotelId)(
      HotelIdWithPrice.byRelevance(hotelRatings).reverse,
      HotelIdWithPrice.byPrice
    )

    var i = 0
    val start = System.currentTimeMillis()
    for {
      e <- iterator
    } {
      i += 1
      val price = new HotelIdWithPrice(e.when, e.nights, e.hotelId, e.minPrice, Pansion.valueOf(e.pansion))
      topHotels += price
      cheapestHotel += price
    }
    WizardTracer.checkpoint("hotels_finish", i)
    log.debug(s"HOTELS: Traversed $i elements in ${System.currentTimeMillis() - start} ms.")

    if (topHotels.size == maxHotelsForDirection && cheapestHotel.nonEmpty) {
      val hotels = cheapestHotel.head :: topHotels.toList
      val list = joinWithHotels(hotels).sortBy(_.minPrice).distinctBy(_.hotel.id).take(maxHotelsForDirection)
      if (list.size == maxHotelsForDirection) {
        Some(ToursResponse(
          request.from,
          request.to.get,
          list
        ))
      } else {
        None
      }
    } else {
      None
    }
  }

  protected def directionBestPrice(direction: Direction, request: ToursWizardRequest, i: AtomicInteger): Option[DirectionWithPrice] = {
    val index = indexHolder.getIndex
    val operatorId = request.operator.getOrElse(0)
    var minPrice = Int.MaxValue
    var when = Int.MaxValue
    var nights = 0
    var j = 0
    for (e <- index.find(operatorId, request.from, WizardIndexing.toDirection(direction.region.id), request.when.from, request.when.until)) {
      j += 1
      if (e.minPrice < minPrice) {
        minPrice = e.minPrice
        when = e.when
        nights = e.nights
      }
    }
    i.addAndGet(j)
    if (minPrice != Int.MaxValue && when != Int.MaxValue) {
      Some(DirectionWithPrice(when, nights, direction, minPrice))
    } else {
      None
    }
  }

  def searchToursDirections(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    val i = new AtomicInteger(0)
    val start = System.currentTimeMillis()

    val toursDirections = directions.all.filter(_.isToursDirection)
    val directionsToScan =
      if (request.markers.contains(NoVisa)) toursDirections.filter(_.noVisa)
      else toursDirections
    val withPrice = {
      for {
        direction <- directionsToScan
        bestPrice <- directionBestPrice(direction, request, i)
      } yield direction -> bestPrice
    }.toMap.seq

    WizardTracer.checkpoint("directions_finish", i.get)
    log.debug(s"DIRECTIONS: Traversed $i elements in ${System.currentTimeMillis() - start} ms.")

    val topDirectionsRaw =
      DirectionsUtils.collectDirections(
        directions = directionsToScan,
        withPrice = withPrice.keySet.map(_.region.id),
        tree = tree,
        count = maxDirections
      )

    val directionsWithPrice = topDirectionsRaw.map(withPrice)

    def topCheapDirections = directionsWithPrice.sortBy(_.minPrice).take(maxDirections)
    def hotDirectionsWithPrice: Vector[DirectionWithPrice] = (for {
      directionWithPrice <- withPrice.values
      if hotDirections.isHot(directionWithPrice.direction.region.id, directionWithPrice.minPrice)
    } yield directionWithPrice) (collection.breakOut)

    val topDirections = if (request.hasHotMarker) {
      val topHot = hotDirectionsWithPrice.sorted(Ordering[DirectionWithPrice].reverse).take(maxDirections)
      if (topHot.size >= maxDirections) topHot
      else (topHot ++ topCheapDirections).distinct.take(maxDirections)
    } else {
      directionsWithPrice.sorted(Ordering[DirectionWithPrice].reverse).take(maxDirections)
    }

    if (topDirections.size == maxDirections) {
      Some(TourDirectionsResponse(
        request.from,
        topDirections
      ))
    } else {
      None
    }
  }
}

object IndexToursSearcher {
  class HotelIdWithPrice(val whenInt: Int, val nights: Int, val hotelId: Int, val minPrice: Int, val pansion: Pansion) {
    def toResponse(hotel: Hotel) = TourWithPrice(whenInt, nights, hotel, minPrice, pansion)
  }
  object HotelIdWithPrice {
    val byPrice = Ordering.by[HotelIdWithPrice, Int](_.minPrice)
    def byRelevance(hotelRatings: HotelRatings) = {
      val ratings = hotelRatings.copy()
      Ordering.by[HotelIdWithPrice, Double](h => ratings.getRelevance(h.hotelId))
    }
  }
}