package ru.yandex.tours.wizard.search

import java.util.concurrent.TimeUnit

import com.google.common.cache.{CacheBuilder, CacheLoader}
import com.google.common.util.concurrent.ListenableFuture
import ru.yandex.tours.direction.{Direction, Directions, DirectionsStats}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelRatings, HotelsIndex}
import ru.yandex.tours.model.direction.Thematics
import ru.yandex.tours.model.filter.HotelFilter
import ru.yandex.tours.model.filter.hotel.{GeoIdFilter, HotelTypeFilter, SearchTypeFilter, StarFilter}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.hotels.HotelsHolder.HotelType
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.query.{HotelMarker, Stars, TourMarker}
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.tours.wizard.WizardTracer
import ru.yandex.tours.wizard.domain._

import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 11.11.15
 */
class RoomsSearcher(hotelsIndex: HotelsIndex,
                    hotelRatings: HotelRatings,
                    tree: region.Tree,
                    geoMappingHolder: GeoMappingHolder,
                    directions: Directions,
                    directionsStats: DirectionsStats)
                   (implicit ec: ExecutionContext) extends WizardToursSearcher {

  private val maxHotelsForDirection = 3
  private val maxDirections = 4

  private val directionsCache = CacheBuilder.newBuilder()
    .refreshAfterWrite(1, TimeUnit.HOURS)
    .build(new CacheLoader[(Set[HotelType], Option[Int]), Seq[Direction]] {
      override def load(params: (Set[HotelType], Option[Int])): Seq[Direction] = {
        val (hotelTypes, regionId) = params
        directions.all
          .filter(_.isRoomsDirection)
          .filterNot(_.isCountry)
          .filter(d => regionId.isEmpty || tree.pathToRoot(d.region).exists(_.id == regionId.get))
          .sorted(relevance(hotelTypes).reverse)
          .take(maxDirections)
      }
      override def reload(key: (Set[HotelType], Option[Int]),
                          oldValue: Seq[Direction]): ListenableFuture[Seq[Direction]] = {
        Future { load(key) }.asGuavaFuture
      }
    })

  def warmUp(): Unit = {
    HotelType.values().foreach { hotelType =>
      directionsCache.get(Set(hotelType) -> None)
    }
  }

  override def search(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    if (request.hotel.isDefined) {
      for {
        hotelId <- request.hotel
        hotel <- hotelsIndex.getHotelById(hotelId)
        response <- searchHotel(request, hotel)
      } yield response
    } else if (request.hasSkiMarker || Regions.isSkyResort(request.to, directions)) {
      searchSkiResorts(request)
    } else if (request.to.isDefined) {
      if (request.to.contains(225)) searchRoomDirections(request)
      else if (geoMappingHolder.isKnownDestination(request.to.get, SearchType.ROOMS)) searchRooms(request)
      else None
    } else if (request.hasHotelMarker || (request.operator.isEmpty && request.markers.contains(TourMarker))) {
      searchRoomDirections(request)
    } else {
      None
    }
  }

  private def searchHotel(request: ToursWizardRequest, hotel: Hotel): Option[ToursWizardResponse] = {
    Some(HotelSnippetResponse(request.from, None, hotel))
  }

  private def hotelsForRequest(request: ToursWizardRequest, additionalFilters: HotelFilter*) = {
    val starsFilter = new StarFilter(request.collectOf[Stars].map(_.count))
    val hotelTypes = request.hotelTypesFilter

    val filters = SearchTypeFilter(SearchType.ROOMS) +: hotelTypes +: starsFilter +: additionalFilters
    val hotels = hotelsIndex.topInRegion(
      request.to.get,
      maxHotelsForDirection,
      filters: _*
    ).toList

    if (hotels.size < maxHotelsForDirection) {
      for {
        region <- tree.region(request.to.get).toList
        hotel <- hotelsIndex.inRectangle(region.boundingBox * 0.66, 30, filters: _*).sorted.reverse
      } yield hotel
    } else {
      hotels
    }
  }

  private def searchRooms(request: ToursWizardRequest): Option[RoomsResponse] = {
    val hotels = hotelsForRequest(request)
    WizardTracer.checkpoint("rooms_finish")
    if (hotels.size < maxHotelsForDirection) Option.empty
    else Some(RoomsResponse(request.from, request.to.get, hotels))
  }

  def searchRoomDirections(request: ToursWizardRequest): Option[RoomDirectionsResponse] = {
    val hotelTypes = request.collectOf[HotelMarker].flatMap(_.hotelType.getSearchTypes)
    val directions = topDirections(hotelTypes, request.to)
    WizardTracer.checkpoint("room_directions_finish")
    Some(RoomDirectionsResponse(request.from, directions))
  }

  def searchSkiResorts(request: ToursWizardRequest): Option[ToursWizardResponse] = {
    val to = request.to
    val exactDirection = to.flatMap(directions.get).filter(_.ski.nonEmpty)
    val skiDirections = directions.all
      .filter(_.ski.nonEmpty)
      .filter(d => to.isEmpty || tree.pathToRoot(d.region).exists(_.id == to.get))

    if (exactDirection.nonEmpty && !request.hasMarker) {
      val hotels = hotelsForRequest(request)

      if (hotels.size < maxHotelsForDirection) Option.empty
      else Some(SkiResortResponse(request.from, request.to.get, exactDirection.get.ski.get, hotels))
    } else if (skiDirections.size >= maxDirections) {
      val sorted = skiDirections
        .sortBy(_.relevance(Thematics.Ski))(Ordering[Double].reverse)

      Some(RoomDirectionsResponse(request.from, sorted))
    } else if (skiDirections.nonEmpty) {
      val geoIdFilter = new GeoIdFilter(skiDirections.map(_.region.id))
      val hotels = hotelsForRequest(request, geoIdFilter)

      if (hotels.size < maxHotelsForDirection) Option.empty
      else Some(RoomsResponse(request.from, request.to.get, hotels))
    } else {
      None
    }
  }

  private def topDirections(hotelTypes: Set[HotelType], regionId: Option[Int]) = {
    directionsCache.get(hotelTypes -> regionId)
  }

  private def relevance(hotelTypes: Set[HotelType]) = {
    Ordering.by[Direction, Double] { direction: Direction =>
      hotelsIndex.count(
        GeoIdFilter(direction.region.id),
        SearchTypeFilter(SearchType.ROOMS),
        HotelTypeFilter(hotelTypes)
      ) * directionsStats.getPriority(direction.region.id, SearchType.ROOMS)
    }
  }
}
