package ru.yandex.tours.wizard.search

import java.io.Closeable

import org.apache.curator.framework.CuratorFramework
import ru.yandex.tours.util.zoo

import scala.util.hashing.MurmurHash3

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.03.15
 */
trait WizardThrottler {
  def setProbability(probability: Double): Unit
  def shouldAnswer(userRequest: String, uid: String): Boolean
}

class ZkWizardThrottler(zkClient: CuratorFramework) extends WizardThrottler with Closeable {

  private val hashRange = Int.MaxValue.toLong - Int.MinValue
  private def longHash(str: String) = MurmurHash3.stringHash(str).toLong & 0xffffffffL

  private val probability = new zoo.SharedValue[Double](zkClient, "/wizard/probability", 1d, zoo.DoubleSerializer)

  def setProbability(probability: Double): Unit = {
    require(probability >= 0d && probability <= 1d, "probability should be in [0, 1] range")
    this.probability.set(probability)
  }

  def shouldAnswer(userRequest: String, uid: String): Boolean = {
    def hash = longHash(uid + "." + userRequest).toDouble
    val prob = probability.get
    prob > 0.99d || (hash / hashRange) < prob
  }

  def close(): Unit = {
    probability.close()
  }
}
