package ru.yandex.tours.wizard.serialize

import org.joda.time.LocalDate
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.direction.Thematics
import ru.yandex.tours.model.filter.hotel.HotelTypeFilter
import ru.yandex.tours.model.hotels.Features._
import ru.yandex.tours.model.hotels.HotelsHolder.HotelType
import ru.yandex.tours.model.hotels.{Address, Hotel, Partners}
import ru.yandex.tours.model.image.ImageFormats
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.search.SearchType.{ROOMS, SearchType, TOURS}
import ru.yandex.tours.model.util.SortType
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.model.wizard.MicroOffer.{HotelMicroOffersProto, MicroOfferProto}
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.query.HotelMarker
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.serialize.json.CommonJsonSerialization
import ru.yandex.tours.util.LabelBuilder
import ru.yandex.tours.util.http.UrlLabelerBuilder
import ru.yandex.tours.util.lang.Dates.RichCompactDate
import ru.yandex.tours.wizard.domain._
import ru.yandex.tours.wizard.experiment.WizardExperiment

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.03.15
 */
trait JsonBuilder {
  def skiResort(req: ToursWizardRequest, response: SkiResortResponse): JSONObject

  def rooms(req: ToursWizardRequest, response: RoomsResponse): JSONObject

  def tours(req: ToursWizardRequest, response: ToursResponse): JSONObject

  def roomDirections(req: ToursWizardRequest, response: RoomDirectionsResponse): JSONObject

  def tourDirections(req: ToursWizardRequest, response: TourDirectionsResponse): JSONObject

  def hotelSnippet(req: ToursWizardRequest, response: HotelSnippetResponse): JSONObject

  def hotelPrices(req: ToursWizardRequest, response: HotelPricesResponse): JSONObject
}

class DefaultJsonBuilder(tree: region.Tree,
                         geoMapping: GeoMappingHolder,
                         searchSettings: SearchSettingsHolder,
                         textBuilder: TextBuilder,
                         urlBuilder: UrlBuilder,
                         redirectUrl: String,
                         labelBuilder: LabelBuilder) extends JsonBuilder {
  private def counter(path: String) = {
    new JSONObject()
      .put("path", path)
  }

  protected def jsonLink(text: String, url: String) = {
    new JSONObject()
      .put("text", text.highlight)
      .put("url", url)
  }

  protected def jsonPrice(minPrice: Int) = {
    new JSONObject()
      .put("minValue", minPrice)
      .put("currency", "RUR")
  }

  protected def rating(from: Int, hotel: Hotel, when: LocalDate, nights: Int)(implicit utm: UtmMark): JSONObject = {
    val json = new JSONObject()
    if (hotel.reviewsCount > 0) {
      json
        .put("count", hotel.reviewsCount)
        .put("link", urlBuilder.forHotelReviews(from, hotel.id, when, nights, utm))
    }
    if (hotel.rating > 0) {
      json.put("value", (hotel.rating * 5).round / 10)
        .put("originalBestValue", "10")
        .put("originalValue", (hotel.rating * 10).round / 10)
    }
    json
  }

  private def hotelInfo(hotel: Hotel): JSONObject = {
    val address = hotel.addresses.getOrElse(Languages.ru,
      if (hotel.addresses.isEmpty) Address.empty(Languages.ru) else hotel.addresses.values.head)

    val res = new JSONObject()
      .put("latitude", hotel.latitude)
      .put("longitude", hotel.longitude)

    res.put("name", textBuilder.shortNameFor(hotel))

    res.put("region", CommonJsonSerialization.regionToJson(hotel.geoId, tree, Languages.ru))

    val mainImage = new JSONObject()
    for (format ← ImageFormats.values) {
      mainImage.put(format.toString, hotel.mainImage.inFormatWithoutProtocol(format))
    }
    res.put("mainImage", mainImage)

    val images = new JSONArray()
    for (image ← hotel.images) {
      images.put(image.baseUrl)
    }
    res.put("images", images)

    if (address.fullAddress.nonEmpty) res.put("address", address.fullAddress)

    if (hotel.phones.nonEmpty) res.put("phone", hotel.phones.head)

    hotel.url.foreach(url ⇒ res.put("hotelUrl", url))
    hotel.partnerIds.filter(_.partner == Partners.backa)
      .foreach(p ⇒ res.append("backaId", p.id))

    if (hotel.`type` != HotelType.UNKNOWN) {
      val obj = new JSONObject()
        .put("code", hotel.`type`.name())
        .put("name", textBuilder.hotelType(hotel.`type`).getOrElse("").capitalize)

      res.put("hotelType", obj)
    }



    val topFeatures = new JSONArray()
    for {
      feature ← hotel.topFeatures
      obj ← convertFeature(feature)
    } topFeatures.put(obj)

    res.put("topFeatures", topFeatures)

    val allFeatures = new JSONArray()
    for {
      feature ← hotel.features
      obj ← convertFeature(feature)
    } allFeatures.put(obj)
    res.put("features", allFeatures)

    res
  }

  private def convertFeature(feature: FeatureValue) = {
    val obj = new JSONObject()
      .put("id", feature.feature.name)

    var ignore = false

    textBuilder.translateName(feature.feature.name) match {
      case Some(name) if name.nonEmpty ⇒
        obj.put("name", name)
      case _ ⇒
        ignore = true
    }

    val translated = textBuilder.translate(feature)
    if (translated.isEmpty) {
      ignore = true
    }


    feature match {
      case EnumFeatureValue(_, Some(value)) ⇒
        obj.put("type", "enum")
          .put("values", translated.asJava)

      case MultipleFeatureValue(_, values) if values.nonEmpty ⇒
        obj.put("type", "enum")
          .put("values", translated.asJava)

      case IntFeatureValue(_, Some(value)) if translated.nonEmpty ⇒
        obj.put("type", "text")
          .put("value", translated.head)

      case BooleanFeatureValue(_, Some(true)) ⇒
        obj.put("type", "bool")
          .put("value", 1)

      case _ ⇒
        ignore = true
    }
    if (ignore) None
    else Some(obj)
  }

  protected def siteLinks(from: Int, hotel: Hotel, when: LocalDate, nights: Int)(implicit utm: UtmMark) = {
    def link(text: String, `type`: String, url: String) = {
      new JSONObject()
        .put("text", text)
        .put("type", `type`)
        .put("url", url)
    }
    val arr = new JSONArray()
    if (hotel.reviewsCount > 0) {
      arr.put(link("Отзывы", "reviews", urlBuilder.forHotelReviews(from, hotel.id, when, nights, utm = utm)))
    }
    arr.put(link("Фото", "photos", urlBuilder.forHotelPhoto(from, hotel.id, when, nights, hotel.mainImage.name, utm = utm)))

    if (hotel.toursSearchAvailable)
      arr.put(link("Туры", "tours", urlBuilder.forHotelAnchor(from, hotel.id, when, nights, "hotel__tours", utm = utm)))

    if (hotel.roomsSearchAvailable)
      arr.put(link("Цены", "prices", urlBuilder.forHotelAnchor(from, hotel.id, when, nights, "hotel__rooms", utm = utm)))

    arr.put(link("На карте", "onMap", urlBuilder.forHotelAnchor(from, hotel.id, when, nights, "hotel__map", utm = utm)))
    arr.put(link("Услуги в отеле", "features", urlBuilder.forHotelAnchor(from, hotel.id, when, nights, "hotel__features", utm = utm)))
  }

  protected val favicon = new JSONObject().put("domain", urlBuilder.faviconDomain)

  implicit class SeqToArray(seq: Seq[AnyRef]) {
    def toJSONArray = seq.foldLeft(new JSONArray()) { _.put(_) }
  }
  implicit class RString(str: String) {
    def highlight = new JSONObject().put("__hl", str)
  }

  protected def basePath(context: SearchType)(implicit utm: UtmMark): JSONArray = {
    new JSONArray()
      .put(jsonLink(textBuilder.serviceName, urlBuilder.forHome(context, utm)))
  }

  protected def pathTo(from: Int, regionId: Int, when: LocalDate, nights: Int,
                       operatorId: Option[Int], context: SearchType)
                      (implicit utm: UtmMark): JSONArray = {
    val region = tree.region(regionId).get
    val path = tree.pathToRoot(region)
      .filter(x => geoMapping.isKnownDestination(x.id) && searchSettings.getRegionSearchSettings(x).isAllowed(context))
      .reverse
    val fixedPath = (path.take(1) ++ path.takeRight(1)).distinct
    val array = basePath(context)
    for (region <- fixedPath) {
      val name = textBuilder.shortNameFor(region)
      val url = urlBuilder.forSearch(from, region.id, when, nights, operatorId, utm = utm, context = context)
      array.put(jsonLink(name, url))
    }
    array
  }

  protected def pathTo(from: Int, hotel: Hotel,
                       when: LocalDate, nights: Int,
                       operatorId: Option[Int],
                       context: SearchType)
                      (implicit utm: UtmMark): JSONArray = {
    val array = pathTo(from, hotel.geoId, when, nights, operatorId, context)
    val name = textBuilder.nameFor(hotel)
    val url = urlBuilder.forHotel(from, hotel.id, when, nights, operatorId, utm, context = context)
    array.put(jsonLink(name, url))
  }

  private def utmFor(req: ToursWizardRequest, response: ToursWizardResponse) = {
    response.wizardType.utm
      .withContent(WizardExperiment.getCurrentExperiment)
      .withTerm(WizardExperiment.getBucketName)
      .withQuery(labelBuilder.encrypt(req.originalRequest))
  }

  def skiResort(req: ToursWizardRequest, response: SkiResortResponse): JSONObject = {
    implicit val utm = utmFor(req, response)
    val context = ROOMS

    val when = req.when.from
    val nights = DefaultRequestGenerator.getDefaultNights(context)
    val region = tree.region(response.to).get

    def hotels = response.results.map { hotel =>
      val url = urlBuilder.forHotel(response.from, hotel.id, when, nights, req.operator, utm, context)
      new JSONObject()
        .put("name", textBuilder.shortNameFor(hotel).highlight)
        .put("url", url)
        .put("reviewsCount", hotel.reviewsCount)
        .put("nights", nights)
        .put("image", hotel.mainImage.inFormatWithoutProtocol(ImageFormats.wiz_3col_4_3))
        .put("text", textBuilder.text(response.to, hotel).highlight)
    }.toJSONArray

    val title = textBuilder.titleForSkiResort(region)
    val url = urlBuilder.forSkiDirection(region.id)
    val snippet = textBuilder.snippetForSkiResort(region, response.ski)

    new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(title, url))
      .put("text", snippet.highlight)
      .put("favicon", favicon)
      .put("path", pathTo(response.from, response.to, when, nights, req.operator, context))
      .put("hotels", hotels)
  }

  def rooms(req: ToursWizardRequest, response: RoomsResponse): JSONObject = {
    implicit val utm = utmFor(req, response)
    val context = ROOMS
    val sortType = if (req.hasCheapMarker) Some(SortType.PRICE_ASC) else None
    val hotelTypes = req.hotelTypes
    val filters = Seq(req.hotelTypesFilter)

    val when = req.when.from
    val nights = DefaultRequestGenerator.getDefaultNights(context)
    val region = tree.region(response.to).get

    def hotels = response.results.map { hotel =>
      val url = urlBuilder.forHotel(response.from, hotel.id, when, nights, req.operator, utm, context)
      new JSONObject()
        .put("name", textBuilder.shortNameFor(hotel).highlight)
        .put("url", url)
        .put("reviewsCount", hotel.reviewsCount)
        .put("nights", nights)
        .put("image", hotel.mainImage.inFormatWithoutProtocol(ImageFormats.wiz_3col_4_3))
        .put("text", textBuilder.text(response.to, hotel).highlight)
    }.toJSONArray

    val title =
      if (req.hasSkiMarker) textBuilder.titleForSkiRooms(region, hotelTypes)
      else textBuilder.titleForRooms(region, hotelTypes, req.operator)
    val url = urlBuilder.forSearch(response.from, response.to, when, nights, req.operator, filters, sortType, utm, context)
    val snippet =
      if (req.hasSkiMarker) textBuilder.snippetForSkiRooms(region, hotelTypes)
      else textBuilder.snippetForRooms(region, hotelTypes, req.operator)

    new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(title, url))
      .put("text", snippet.highlight)
      .put("favicon", favicon)
      .put("path", pathTo(response.from, response.to, when, nights, req.operator, context))
      .put("hotels", hotels)
  }

  def tours(req: ToursWizardRequest, response: ToursResponse): JSONObject = {
    implicit val utm = utmFor(req, response)
    val context = TOURS
    val sortType = if (req.hasCheapMarker) Some(SortType.PRICE_ASC) else None

    def hotels = response.results.map { item =>
      val hotel = item.hotel
      val url = urlBuilder.forHotel(response.from, hotel.id, item.when, item.nights, req.operator, utm, context)
      new JSONObject()
        .put("name", textBuilder.shortNameFor(hotel).highlight)
        .put("url", url)
        .put("reviewsCount", hotel.reviewsCount)
        .put("nights", item.nights)
        .put("image", item.hotel.mainImage.inFormatWithoutProtocol(ImageFormats.wiz_3col_4_3))
        .put("text", textBuilder.text(item).highlight)
        .put("price", jsonPrice(item.minPrice))
    }.toJSONArray

    val minimal = response.results.minBy(_.minPrice)
    val url = urlBuilder.forSearch(response.from, response.to, minimal.when, minimal.nights, req.operator, Seq.empty, sortType, utm, context)

    val region = tree.region(response.to).get

    new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(textBuilder.titleFor(response.from, region, req.operator, req.hasHotMarker), url))
      .put("text", textBuilder.snippetFor(region, req.operator).highlight)
      .put("favicon", favicon)
      .put("path", pathTo(response.from, response.to, minimal.when, minimal.nights, req.operator, context))
      .put("hotels", hotels)
  }

  def roomDirections(req: ToursWizardRequest, response: RoomDirectionsResponse): JSONObject = {
    implicit val utm = utmFor(req, response)
    val sortType = if (req.hasCheapMarker) Some(SortType.PRICE_ASC) else None
    val context = ROOMS
    val hotelTypes = req.collectOf[HotelMarker].map(_.hotelType)
    val filters = Seq(
      HotelTypeFilter.construct(hotelTypes.flatMap(_.getSearchTypes).toSeq)
    )

    val when = req.when.from
    val nights = DefaultRequestGenerator.getDefaultNights(context)

    def directions = response.directions.map { direction =>
      val url =
        if (req.hasSkiMarker) urlBuilder.forSkiDirection(direction.region.id)
        else urlBuilder.forSearch(response.from, direction.region.id, when, nights, req.operator, filters, sortType, utm, context)
      new JSONObject()
        .put("name", textBuilder.shortNameFor(direction.region).highlight)
        .put("image", direction.mainImage.inFormatWithoutProtocol(ImageFormats.wiz_4col_4_3))
        .put("url", url)
    }.toJSONArray

    val title =
      if (req.hasSkiMarker) textBuilder.roomsSkiTitle(req.to)
      else textBuilder.roomsTitle(hotelTypes, req.operator)

    val titleUrl =
      if (req.hasSkiMarker) {
        val filters = Map("country" -> req.to.fold("")(_.toString))
        urlBuilder.forThematic(Thematics.Ski, context, filters, utm)
      } else {
        urlBuilder.forHome(context, utm)
      }

    val snippet =
      if (req.hasSkiMarker) textBuilder.roomsSkiSnippet(req.to)
      else textBuilder.roomsSnippet(hotelTypes, req.operator)

    new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(title, titleUrl))
      .put("text", snippet.highlight)
      .put("favicon", favicon)
      .put("path", basePath(context))
      .put("directions", directions)
  }

  def tourDirections(req: ToursWizardRequest, response: TourDirectionsResponse): JSONObject = {
    val sortType = Some(SortType.PRICE_ASC)
    val context = TOURS
    implicit val utm = utmFor(req, response)

    def directions = response.directions.map { item =>
      val url = urlBuilder.forSearch(response.from, item.direction.region.id,
        item.when, item.nights, req.operator, filters = Seq.empty, sortType, utm, context)
      new JSONObject()
        .put("name", textBuilder.shortNameFor(item.direction.region).highlight)
        .put("image", item.direction.mainImage.inFormatWithoutProtocol(ImageFormats.wiz_4col_4_3))
        .put("url", url)
        .put("price", jsonPrice(item.minPrice))
    }.toJSONArray

    new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(textBuilder.toursTitle(response.from, req.operator, req.hasHotMarker), urlBuilder.forHome(context, utm)))
      .put("text", textBuilder.toursSnippet(req.operator).highlight)
      .put("favicon", favicon)
      .put("path", basePath(context))
      .put("directions", directions)
  }

  def hotelSnippet(req: ToursWizardRequest, response: HotelSnippetResponse) = {
    implicit val utm = utmFor(req, response)
    val hotel = response.hotel
    val context = if (hotel.toursSearchAvailable) TOURS else ROOMS

    val bestPrice = response.bestPrice
    val when = bestPrice.fold(req.when.from)(_.when)
    val nights = bestPrice.fold(DefaultRequestGenerator.getDefaultNights(context))(_.nights)
    val url = urlBuilder.forHotel(response.from, hotel.id, when, nights, utm = utm, context = context)
    val offers = response.offers.flatMap(hotelOffers(_, req.from, req.operator))

    val json = new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(textBuilder.titleFor(response.from, hotel, tourMarker = false, req.hasHotMarker), url))
      .put("text", textBuilder.snippetFor(hotel).highlight)
      .put("favicon", favicon)
      .put("path", pathTo(response.from, hotel, when, nights, operatorId = None, context))
      .put("rating", rating(response.from, hotel, when, nights))
      .put("sitelinks", siteLinks(response.from, hotel, when, nights))
      .put("hotel", hotelInfo(hotel))
    offers.foreach(json.put("offers", _))
    json
  }

  def hotelPrices(req: ToursWizardRequest, response: HotelPricesResponse): JSONObject = {
    implicit val utm = utmFor(req, response)
    val hotel = response.hotel
    val context = if (hotel.toursSearchAvailable) TOURS else ROOMS

    def prices = response.prices.map { price =>
      val url = urlBuilder.forHotel(response.from, response.hotel.id, price.when, price.nights, req.operator, utm, context)
      new JSONObject()
        .put("name", textBuilder.nights(price.nights))
        .put("nights", price.nights)
        .put("url", url)
        .put("price", jsonPrice(price.minPrice))
    }.toJSONArray

    val minimal = response.prices.minBy(_.minPrice)
    val url = urlBuilder.forHotel(response.from, hotel.id, minimal.when, minimal.nights, req.operator, utm, context)
    val offers = response.offers.flatMap(hotelOffers(_, req.from, req.operator))

    val json = new JSONObject()
      .put("type", response.wizardType)
      .put("counter", counter(response.wizardType.getCounter))
      .put("title", jsonLink(textBuilder.titleFor(response.from, hotel, tourMarker = true, req.hasHotMarker), url))
      .put("text", textBuilder.hotelPricesText(response.from).highlight)
      .put("favicon", favicon)
      .put("path", pathTo(response.from, hotel, minimal.when, minimal.nights, req.operator, context))
      .put("prices", prices)
      .put("rating", rating(response.from, hotel, minimal.when, minimal.nights))
      .put("hotel", hotelInfo(hotel))
    offers.foreach(json.put("offers", _))
    json
  }

  private def hotelOffers(hotelOffers: HotelMicroOffersProto,
                          from: Int,
                          operator: Option[Int])
                         (implicit utm: UtmMark): Option[JSONObject] = {
    val hotelId = hotelOffers.getHotelId
    val user = UserIdentifiers(None, Some(WizardExperiment.getUid), None)

    val offers: scala.collection.mutable.Buffer[JSONObject] = for {
      offer <- hotelOffers.getOffersList.asScala
      provider <- textBuilder.providerNameOpt(offer.getOperatorId)
      pansion <- textBuilder.translate(offer.getPansion)
    } yield {
      // hack to recover correct label name (which is special only for ostrovok)
      val partnerId = if (offer.getUrl.contains("ostrovok.ru")) Partners.ostrovok.id else Partners.unknown.id
      val labeler = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder)
        .buildFrom(hotelId, offer.getOperatorId, partnerId, offer.getPrice)
      new JSONObject()
        .put("provider", provider)
        .put("bookingUrl", labeler(offer.getUrl))
        .put("room", offer.getRoomName)
        .put("pansion", pansion)
        .put("price", jsonPrice(offer.getPrice))
    }

    if (offers.isEmpty) None
    else {
      val when = hotelOffers.getCompactWhen.toLocalDate
      val nights = hotelOffers.getNights
      val url = urlBuilder.forHotel(from, hotelId, when, nights, operator, utm, SearchType.ROOMS)

      Some(new JSONObject()
        .put("offerList", offers.toJSONArray)
        .put("searchUrl", url)
        .put("when", when)
        .put("nights", nights)
      )
    }
  }
}
