package ru.yandex.tours.wizard.serialize;

import NMetaProtocol.Meta.*;
import com.google.protobuf.ByteString;
import com.google.protobuf.TextFormat;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import scala.Tuple2;

import java.io.UnsupportedEncodingException;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import static com.google.protobuf.ByteString.copyFrom;

/**
 * User: daedra
 * Date: 10.04.13
 * Time: 20:32
 */
public class ReportProtobufSerializer {
    // In libyandex-maps-protosearch SEGMENT_ID = "xcript"
    private static final String SEGMENT_ID = "commonProtobuf";
    private static final String BLENDER_RELEVANCE = "blender_relev";
    private static final String BLENDER_FACTORS = "blender_factors";
    private static final int VERSION = 1;
//    private static final float DOC_COUNT_MULTIPLIER = 1.0f;

    private static final int MAGIC_RELEVANCE_STAT = 1;
    private static final int MAGIC_MODE = 1;


    // XXX: must be set to service specific name, because this is checked in SERP
    private String source;
    private String encoding;
    private String grouping;
    private String url;

    public byte[] serialize(String document, String info, double relevance, String factors, boolean humanReadable) throws UnsupportedEncodingException {
        return serialize(Collections.singletonList(Tuple2.apply(document, info)), relevance, factors, humanReadable);
    }

    public byte[] serialize(List<Tuple2<String, String>> documents, double relevance, String factors, boolean humanReadable) {
        try {
            TReport report = createReport(documents, relevance, factors);

            if (humanReadable) {
                String result = TextFormat.printToUnicodeString(report);
                return result.getBytes("UTF-8");
            } else {
                return report.toByteArray();
            }
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unable to serialize to protobuf", e);
        }
    }

    public byte[] serializeError(String errorMessage, boolean humanReadable) {
        try {
            TReport report = createErrorReport(errorMessage);


            if (humanReadable) {
                String result = TextFormat.printToUnicodeString(report);
                return result.getBytes("UTF-8");
            } else {
                return report.toByteArray();
            }

        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unalbe to serialize to protobuf", e);
        }
    }

    private TReport createErrorReport(String text) throws UnsupportedEncodingException {
        TReport.Builder reportBuilder = TReport.newBuilder();

        THead.Builder headBuilder = THead.newBuilder();
//        headBuilder.setDocCountMultiplier(DOC_COUNT_MULTIPLIER);
        headBuilder.setVersion(VERSION);
        reportBuilder.setHead(headBuilder.build());

        TErrorInfo.Builder errorInfo = TErrorInfo.newBuilder();
        errorInfo.setGotError(TErrorInfo.TGotError.YES);
        errorInfo.setText(copyFrom(StringUtils.trimToEmpty(text), encoding));
        reportBuilder.setErrorInfo(errorInfo.build());
        return reportBuilder.build();
    }

    private TPairBytesBytes pair(String key, String value) {
        return TPairBytesBytes.newBuilder()
                .setKey(ByteString.copyFromUtf8(key))
                .setValue(ByteString.copyFromUtf8(value))
                .build();
    }

    private TReport createReport(List<Tuple2<String, String>> documents, double relev, String factors) throws UnsupportedEncodingException {
        if (documents == null) {
            documents = Collections.emptyList();
        }
        TReport.Builder reportBuilder = TReport.newBuilder();

        TErrorInfo errorInfo = TErrorInfo.newBuilder().setGotError(TErrorInfo.TGotError.NO).build();
        reportBuilder.setErrorInfo(errorInfo);
        TDebugInfo debugInfo = TDebugInfo.newBuilder().setAnswerIsComplete(true).build();
        reportBuilder.setDebugInfo(debugInfo);

        THead.Builder headBuilder = THead.newBuilder();
//        headBuilder.setDocCountMultiplier(DOC_COUNT_MULTIPLIER);
        headBuilder.setVersion(VERSION);
        reportBuilder.setHead(headBuilder.build());

        // NOTICE: repeats three times, may be works well without it
        reportBuilder.addTotalDocCount(documents.size());
        reportBuilder.addTotalDocCount(documents.size());
        reportBuilder.addTotalDocCount(documents.size());

        reportBuilder.addSearcherProp(pair(BLENDER_RELEVANCE, String.format(Locale.ENGLISH, "%.2f", relev)));
        reportBuilder.addSearcherProp(pair(BLENDER_FACTORS, factors));

        TGrouping.Builder groupingBuilder = TGrouping.newBuilder();
        if (!isSourceEmpty(source)) {
            groupingBuilder.setIsFlat(TGrouping.TIsFlat.NO);
            groupingBuilder.setAttr(copyFrom(grouping, encoding));
            groupingBuilder.setMode(MAGIC_MODE);
        } else {
            groupingBuilder.setIsFlat(TGrouping.TIsFlat.YES);
        }

        // NOTICE: repeats three times, may be works well without it
        groupingBuilder.addNumDocs(documents.size());
        groupingBuilder.addNumDocs(documents.size());
        groupingBuilder.addNumDocs(documents.size());


        int documentCounter = 0;
        for (Tuple2<String, String> dataWithInfo : documents) {
            final int relevance = documents.size() - documentCounter;

            TGroup.Builder groupBuilder = TGroup.newBuilder();
            groupBuilder.setRelevance(relevance);
            groupBuilder.setCategoryName(copyFrom(String.valueOf(documentCounter), encoding));

            // NOTICE: repeats three times, may be works well without it
            groupBuilder.addRelevStat(MAGIC_RELEVANCE_STAT);
            groupBuilder.addRelevStat(MAGIC_RELEVANCE_STAT);
            groupBuilder.addRelevStat(MAGIC_RELEVANCE_STAT);

            TDocument.Builder documentBuilder = TDocument.newBuilder();
            documentBuilder.setRelevance(relevance);

            TArchiveInfo.Builder archiveInfoBuilder = TArchiveInfo.newBuilder();
            archiveInfoBuilder.addPassage(copyFrom(getPassage(dataWithInfo._1()), encoding));

            //SERP DATA
            TPairBytesBytes.Builder gtaRelatedAttributeSerpDataBuilder = TPairBytesBytes.newBuilder();
            gtaRelatedAttributeSerpDataBuilder.setKey(copyFrom("_SerpData", encoding));
            gtaRelatedAttributeSerpDataBuilder.setValue(copyFrom(dataWithInfo._1(), encoding));

            //SERP INFO
            TPairBytesBytes.Builder gtaRelatedAttributeSerpInfoBuilder = TPairBytesBytes.newBuilder();
            gtaRelatedAttributeSerpInfoBuilder.setKey(copyFrom("_SerpInfo", encoding));
            gtaRelatedAttributeSerpInfoBuilder.setValue(copyFrom(dataWithInfo._2(), encoding));

            archiveInfoBuilder.addGtaRelatedAttribute(gtaRelatedAttributeSerpDataBuilder.build());
            archiveInfoBuilder.addGtaRelatedAttribute(gtaRelatedAttributeSerpInfoBuilder.build());

            if (url != null) {
                archiveInfoBuilder.setUrl(copyFrom(url, encoding));
            }
            documentBuilder.setArchiveInfo(archiveInfoBuilder.build());

            groupBuilder.addDocument(documentBuilder.build());
            groupingBuilder.addGroup(groupBuilder.build());
            documentCounter++;
        }

        // NOTICE: repeats three times, may be works well without it
        groupingBuilder.addNumGroups(documents.size());
        groupingBuilder.addNumGroups(documents.size());
        groupingBuilder.addNumGroups(documents.size());

        reportBuilder.addGrouping(groupingBuilder.build());

        return reportBuilder.build();
    }

    private boolean isSourceEmpty(String source) {
        return source == null || source.isEmpty();
    }

    public void setSource(String source) {
        this.source = source;
    }

    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    public void setGrouping(String grouping) {
        this.grouping = grouping;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    private String getPassage(String doc) {
        String md5 = DigestUtils.md5Hex(doc);
        while (md5.length() < 48)
            md5 = md5 + md5;
        return md5;
    }
}
