package ru.yandex.tours.wizard.serialize

import java.util.regex.Pattern

import ru.yandex.tours.filter.holder.hotel.WifiFilterHolder
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Features._
import ru.yandex.tours.model.hotels.HotelsHolder.HotelType
import ru.yandex.tours.model.hotels.{Features, Hotel}
import ru.yandex.tours.operators.{HotelProviders, TourOperators}
import ru.yandex.tours.query.QueryHotelType
import ru.yandex.tours.resorts.Ski
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.tanker.Translations
import ru.yandex.tours.util.Speller
import ru.yandex.tours.util.naming.HotelNameUtils
import ru.yandex.tours.wizard.domain.TourWithPrice

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.02.15
 *
 * TODO move texts to tanker
 */
class TextBuilder(tree: region.Tree,
                  geoMapping: GeoMappingHolder,
                  searchSettings: SearchSettingsHolder,
                  translations: Translations,
                  tourOperators: TourOperators,
                  hotelProviders: HotelProviders) {

  // HOTELS-215
  private val shortNames = Map(
    210 -> "ОАЭ",
    20917 -> "Доминикана",
    10022 -> "Сейшелы",
    1000000144 -> "Газпром"
  )

  private val shortAccusative = Map(
    210 -> "ОАЭ",
    20917 -> "Доминикану",
    10022 -> "Сейшелы"
  )

  private val hotelTypePattern = Pattern.compile("(отель|пансион|гостиница|гостевой дом|хостел)", Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE)

  val lang = Languages.ru

  val serviceName = "Яндекс.Путешествия"

  def roomsTitle(hotelTypes: Set[QueryHotelType], operatorId: Option[Int]) = {
    val hotelType = hotelTypes.collectFirst {
      case QueryHotelType.RU_HOTEL => "гостиниц"
      case QueryHotelType.APARTMENTS => "апартаментов"
      case QueryHotelType.SANATORIUM => "санаториев"
      case QueryHotelType.HOSTEL => "хостелов"
      case QueryHotelType.GUESTHOUSE => "гостевых домов"
      case QueryHotelType.COTTAGE => "коттеджей и гостевых домов"
      case QueryHotelType.VILLA => "вилл"
      case QueryHotelType.MOTEL => "мотелей"
      case QueryHotelType.APARTHOTEL => "апарт-отелей"
      case QueryHotelType.CHALET => "шале"
      case QueryHotelType.MINI_HOTEL => "мини-отелей"
      case QueryHotelType.BOARDING_HOUSE => "пансионатов"
      case QueryHotelType.HOLIDAY_CAMP => "баз отдыха"
      case QueryHotelType.HOLIDAY_HOUSE => "домов отдыха, санаториев и пансионатов"
    }.getOrElse("отелей")
    s"Поиск $hotelType. Сравнение цен от ведущих систем бронирования отелей"
  }

  def roomsSkiTitle(to: Option[Int]) = {
    to.flatMap(tree.region) match {
      case Some(region) => s"Популярные горнолыжные курорты ${region.genitive} на Яндекс.Путешествиях. Сравнение цен от ведущих систем бронирования отелей"
      case None => "Популярные горнолыжные курорты на Яндекс.Путешествиях. Сравнение цен от ведущих систем бронирования отелей"
    }
  }

  def toursTitle(fromRegion: Int, operatorId: Option[Int], hotMarker: Boolean) = {
    val hotTours = if (hotMarker) "горящих " else ""
    if (operatorId.isEmpty)
      "Поиск " + hotTours + "туров " + from(fromRegion) + ". Сравнение цен от ведущих туроператоров"
    else
      "Поиск и покупка " + hotTours + "туров " + from(fromRegion) + byOperator(operatorId)
  }

  def roomsSnippet(hotelTypes: Set[QueryHotelType], operatorId: Option[Int]): String = {
    val (first, second) = hotelTypes.collectFirst {
      case QueryHotelType.RU_HOTEL => "на гостиницы по всем миру" -> "о гостиницах"
      case QueryHotelType.APARTMENTS => "на апартаменты по всем миру" -> "об апартаментах"
      case QueryHotelType.SANATORIUM => "на санатории" -> "о санаториях"
      case QueryHotelType.HOSTEL => "на хостелы" -> "о хостелах"
      case QueryHotelType.GUESTHOUSE => "на гостевые дома" -> "о гостевых домах"
      case QueryHotelType.COTTAGE => "на коттеджи и гостевые дома" -> "о коттеджах"
      case QueryHotelType.VILLA => "на виллы" -> "о виллах"
      case QueryHotelType.MOTEL => "на мотели" -> "о мотелях"
      case QueryHotelType.APARTHOTEL => "на апарт-отели" -> "об апарт-отелях"
      case QueryHotelType.CHALET => "на шале" -> "о шале"
      case QueryHotelType.MINI_HOTEL => "на мини-отели" -> "о мини-отелях"
      case QueryHotelType.BOARDING_HOUSE => "на пансионаты" -> "о пансионатах"
      case QueryHotelType.HOLIDAY_CAMP => "на базы отдыха" -> "о базах отдыха"
      case QueryHotelType.HOLIDAY_HOUSE => "на дома отдыха, санатории и пансионаты" -> "о домах отдыха"
    }.getOrElse("на отели по всем миру" -> "об отелях")
    s"Сравнение цен $first от систем онлайн-бронирования отелей. Подробная информация $second: описание, фотографии, рейтинг и отзывы."
  }

  def roomsSkiSnippet(to: Option[Int]): String = {
    to.flatMap(tree.region) match {
      case Some(region) => s"Информация о горнолыжных курортах ${region.genitive}: описание, фотографии, рейтинг, отзывы и цены."
      case None => s"Информация о горнолыжных курортах: описание, фотографии, рейтинг, отзывы и цены."
    }
  }

  def toursSnippet(operatorId: Option[Int]): String = {
    val suffix = " Подробная информация об отелях: описание, фотографии, рейтинг и отзывы."
    operatorId.flatMap(tourOperators.getById) match {
      case Some(operator) => s"Туры от ${operator.name} и других туроперторов. $suffix"
      case _ => s"Выгодные предложения проверенных туроператоров и систем бронирования отелей. $suffix"
    }
  }
  def hotelPricesText(fromRegion: Int): String =
    s"Цены с перёлетом ${from(fromRegion)} на 2 взрослых:"

  def translate(pansion: Pansion): Option[String] = {
    translations.translate("common.blocks:i-pansions-label", pansion.name, lang)
  }

  def translateName(featureName: String): Option[String] = {
    translations.translateFeature(featureName, lang)
  }

  def translate(feature: FeatureValue): Seq[String] = {
    def translate(keySet: String, value: String): Seq[String] =
      translations.translate(keySet, "full_" + value, lang) //hack
        .orElse(translations.translate(keySet, value, lang))
        .toSeq

    feature match {
      case BooleanFeatureValue(_, Some(true)) ⇒
        translate("common.blocks:features__decl_type_hotel", feature.feature.name)
      case IntFeatureValue(_, Some(i)) =>
        Seq.empty
      case EnumFeatureValue(_, Some(value)) ⇒
        val keySet = "common.blocks:i-" + feature.feature.name.replace("_", "-") + "-label"
        translate(keySet, value)
      case MultipleFeatureValue(_, values) ⇒
        val keySet = "common.blocks:i-" + feature.feature.name.replace("_", "-") + "-label"
        values.flatMap { value =>
          translate(keySet, value)
        }.toSeq
      case _ ⇒ Seq.empty
    }
  }

  def nights(n: Int): String = Speller.nights(n)

  def byOperator(operatorId: Option[Int]): String = {
    operatorId.flatMap(tourOperators.getById).fold("")(o => " от " + o.name)
  }

  def providerNameOpt(providerId: Int): Option[String] = {
    hotelProviders.getById(providerId).map(_.name)
  }

  def nameFor(hotel: Hotel): String = hotel.name(Languages.ru)
  def nameFor(prefix: String, hotel: Hotel): String = {
    val name = hotel.name(Languages.ru)
    if (HotelNameUtils.hasHotelPrefix(name)) name
    else prefix + name
  }

  def starsFor(hotel: Hotel): String = if (hotel.star.id != 0) " " + hotel.star.id + "*" else ""

  def pathFor(hotel: Hotel): Seq[String] = {
    for {
      hotelRegion <- tree.region(hotel.geoId).toSeq
      region <- tree.pathToRoot(hotelRegion)
        .filter(x => geoMapping.isKnownDestination(x.id) && searchSettings.getRegionSearchSettings(x).isSearchable)
        .reverse
    } yield nameFor(region)
  }

  def shortNameFor(hotel: Hotel): String = {
    nameFor(hotel) + starsFor(hotel)
  }

  def fullNameFor(prefix: String, hotel: Hotel): String = {
    nameFor(prefix, hotel) + starsFor(hotel) + locationFor(hotel)
  }

  def locationFor(hotel: Hotel) = {
    val path = pathFor(hotel)
    if (path.isEmpty) ""
    else {
      (path.take(1) ++ path.takeRight(1)).distinct.mkString(" (", ", ", ")")
    }
  }

  def titleFor(fromRegion: Int, hotel: Hotel, tourMarker: Boolean, hotMarker: Boolean): String = {
    if (tourMarker) {
      val tours = if (hotMarker) "Горящие туры" else "Туры"
      s"$tours в ${fullNameFor("отель ", hotel)} ${from(fromRegion)}"
    } else
      s"${fullNameFor("Отель ", hotel)} — отзывы, фото, цены"
  }

  def snippetFor(hotel: Hotel): String = {
    val hotelName = nameFor(hotel)
    val pricesText = if (hotel.toursSearchAvailable) "цены на туры" else "сравнение цен от систем онлайн-бронирования"
    s"Отзывы, фото, $pricesText, подробная информация об отеле, интересные факты и рейтинг — мнения туристов об отеле $hotelName"
  }

  def text(to: Int, hotel: Hotel): String = {
    val reviews = if (hotel.reviewsCount > 0) Some(Speller.reviews(hotel.reviewsCount)) else Option.empty
    val regionName = if (hotel.geoId != to) tree.region(hotel.geoId).map(shortNameFor) else Option.empty
    val freeWiFi = if (WifiFilterHolder.getValues(hotel).nonEmpty) Some("бесплатный Wi-Fi") else Option.empty
    (reviews ++ freeWiFi ++ regionName).mkString(", ").capitalize
  }

  def text(hotelPrice: TourWithPrice): String = {
    val regionName = tree.region(hotelPrice.hotel.geoId).map(_.name(Languages.ru))
    val pansion = translate(hotelPrice.pansion)
    (Seq(nights(hotelPrice.nights))
      ++ pansion
      ++ regionName).mkString(", ")
  }

  def features(hotel: Hotel, count: Int = 3): Seq[String] = {
    val hotelFeatures = hotel.features.map(f => f.feature.name -> f).toMap

    val translations = for {
      name <- Features.topFeatures.toIterator
      feature <- hotelFeatures.get(name).toSeq
      translation <- translate(feature)
    } yield translation

    translations.filter(_.nonEmpty).take(count).zipWithIndex.map {
      case (value, 0) => value
      case (value, _) =>
        val arr = value.toCharArray
        arr(0) = arr(0).toLower
        new String(arr)
    }.toSeq
  }

  def hotelType(hotelType: HotelType): Option[String] = {
    translations.translate(hotelType, lang)
  }

  def nameFor(regionId: Int): String = tree.region(regionId).fold("")(nameFor)
  def nameFor(region: Region): String = region.name(Languages.ru)

  def shortNameFor(regionId: Int): String = shortNames.getOrElse(regionId, nameFor(regionId))
  def shortNameFor(region: Region): String = shortNames.getOrElse(region.id, nameFor(region))

  def shortAccusativeFor(region: Region): String = shortAccusative.getOrElse(region.id, region.accusative)

  def from(regionId: Int): String = tree.region(regionId).fold("")(from)
  def from(region: Region): String = "из " + region.genitive


  private def roomsTitlePrefix(hotelTypes: Set[QueryHotelType]) = hotelTypes.collectFirst {
    case QueryHotelType.RU_HOTEL => "Гостиницы"
    case QueryHotelType.APARTMENTS => "Апартаменты"
    case QueryHotelType.SANATORIUM => "Санатории"
    case QueryHotelType.HOSTEL => "Хостелы"
    case QueryHotelType.GUESTHOUSE => "Гостевые дома"
    case QueryHotelType.COTTAGE => "Коттеджи и гостевые дома"
    case QueryHotelType.VILLA => "Виллы"
    case QueryHotelType.MOTEL => "Мотели"
    case QueryHotelType.APARTHOTEL => "Апарт-отели"
    case QueryHotelType.CHALET => "Шале"
    case QueryHotelType.MINI_HOTEL => "Мини-отели"
    case QueryHotelType.BOARDING_HOUSE => "Пансионаты"
    case QueryHotelType.HOLIDAY_CAMP => "Базы отдыха"
    case QueryHotelType.HOLIDAY_HOUSE => "Дома отдыха, санатории и пансионаты"
  }.getOrElse("Отели")

  def titleForRooms(region: Region, hotelTypes: Set[QueryHotelType], providerId: Option[Int]): String = {
    val prefix = roomsTitlePrefix(hotelTypes)
    val country = tree.country(region).filter(_ != region).fold("")(r => " (" + nameFor(r) + ")")
    s"$prefix ${region.genitive}$country. Сравнение цен от ведущих систем бронирования отелей"
  }

  def titleForSkiResort(region: Region): String = {
    val name = shortNameFor(region)
    val country = tree.country(region).filter(_ != region).fold("")(r => " (" + nameFor(r) + ")")
    s"Горнолыжный курорт $name$country на Яндекс.Путешествиях"
  }

  def titleForSkiRooms(region: Region, hotelTypes: Set[QueryHotelType]): String = {
    val prefix = roomsTitlePrefix(hotelTypes)
    val country = tree.country(region).filter(_ != region).fold("")(r => " (" + nameFor(r) + ")")
    s"$prefix для горнолыжного отдыха ${region.preposition} ${region.locative}$country на Яндекс.Путешествиях"
  }

  def titleFor(fromRegion: Int, region: Region, operatorId: Option[Int], hotMarker: Boolean): String = {
    val tours = if (hotMarker) "Горящие туры" else "Туры"
    if (operatorId.isEmpty)
      s"$tours ${region.preposition} ${shortAccusativeFor(region)} ${from(fromRegion)}. Цены на отдых от ведущих туроператоров"
    else
      s"$tours ${region.preposition} ${shortAccusativeFor(region)} ${from(fromRegion)}${byOperator(operatorId)}"
  }

  private def roomsSnippetPrefix(hotelTypes: Set[QueryHotelType]) = hotelTypes.collectFirst {
    case QueryHotelType.RU_HOTEL => "подходящую гостиницу" -> "на гостиницы"
    case QueryHotelType.APARTMENTS => "апартаменты" -> "на апартаменты"
    case QueryHotelType.SANATORIUM => "санаторий" -> "на санатории"
    case QueryHotelType.HOSTEL => "хостел" -> "на хостелы"
    case QueryHotelType.GUESTHOUSE => "гостевой дом" -> "на гостевые дома"
    case QueryHotelType.COTTAGE => "коттедж" -> "на коттеджи и гостевые дома"
    case QueryHotelType.VILLA => "виллу" -> "на виллы"
    case QueryHotelType.MOTEL => "мотель" -> "на мотели"
    case QueryHotelType.APARTHOTEL => "апарт-отель" -> "на апарт-отели"
    case QueryHotelType.CHALET => "шале" -> "на шале"
    case QueryHotelType.MINI_HOTEL => "мини-отель" -> "на мини-отели"
    case QueryHotelType.BOARDING_HOUSE => "пансионат" -> "на пансионаты"
    case QueryHotelType.HOLIDAY_CAMP => "базу отдыха" -> "на базы отдыха"
    case QueryHotelType.HOLIDAY_HOUSE => "дом отдыха" -> "на базы отдыха, санатории и пансионаты"
  }.getOrElse("подходящий отель" -> "на отели")

  def snippetForRooms(region: Region, hotelTypes: Set[QueryHotelType], operatorId: Option[Int]): String = {
    val (first, second) = roomsSnippetPrefix(hotelTypes)
    s"Выбирайте $first по описанию, фотографиям, рейтингу и отзывам. Сравнение цен $second ${region.genitive} от систем онлайн-бронирования отелей."
  }

  def snippetForSkiResort(region: Region, ski: Ski): String = {
    val name = shortNameFor(region)
    val sb = new StringBuilder
    sb ++= "Читайте информацию о горнолыжном курорте "
    sb ++= name
    sb ++= ": карты склонов, "
    if (ski.skiPassInfo.nonEmpty) sb ++= "цены на ски-пассы, "
    sb ++= "фотографии, отзывы. Выбор вариантов размещения от проверенных туроператоров и систем онлайн-бронирования отелей."
    sb.toString()
  }

  def snippetForSkiRooms(region: Region, hotelTypes: Set[QueryHotelType]): String = {
    val (first, second) = roomsSnippetPrefix(hotelTypes)
    val country = tree.country(region).filter(_ != region).fold("")(r => " (" + nameFor(r) + ")")
    s"Выбирайте $first рядом с горнолыжными курортами ${region.genitive}$country. Сравнение цен $second от систем онлайн-бронирования отелей."
  }

  def snippetFor(region: Region, operatorId: Option[Int]): String = {
    val prefix = s"Выбирайте подходящий отель по описанию, фотографиям, рейтингу и отзывам. Туры ${region.preposition} ${region.accusative} "
    operatorId.flatMap(tourOperators.getById) match {
      case Some(operator) => s"$prefix от ${operator.name} и других туроперторов."
      case _ => s"$prefix от проверенных туроператоров и онлайн-турагентств."
    }
  }
}
