package ru.yandex.tours.wizard.serialize

import org.json.JSONObject
import ru.yandex.tours.query._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.wizard.domain._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.02.15
 */
class WizardResponseSerializer(protobufSerializer: ReportProtobufSerializer, jsonBuilder: JsonBuilder) extends Logging {

  private val serpInfo = {
    new JSONObject()
      .put("type", "construct")
      .put("format", "json")
      .toString
  }

  def serialize(request: ToursWizardRequest, response: ToursWizardResponse, humanReadable: Boolean): Array[Byte] = {
    val data = response match {
      case ski: SkiResortResponse => jsonBuilder.skiResort(request, ski)
      case rr: RoomsResponse => jsonBuilder.rooms(request, rr)
      case hr: ToursResponse => jsonBuilder.tours(request, hr)
      case rdr: RoomDirectionsResponse => jsonBuilder.roomDirections(request, rdr)
      case tdr: TourDirectionsResponse => jsonBuilder.tourDirections(request, tdr)
      case hpr: HotelPricesResponse => jsonBuilder.hotelPrices(request, hpr)
      case hs: HotelSnippetResponse => jsonBuilder.hotelSnippet(request, hs)
    }

    // https://wiki.yandex-team.ru/tourism/201415/development/wizard#faktory
    val factors = Seq(
      request.from == request.userRegion,
      request.hotel.isDefined,
      request.to.isDefined,
      request.operator.isDefined,
      request.markers.contains(TourMarker),
      request.markers.contains(PriceMarker),
      request.markers.contains(OfficialSiteMarker),
      request.markers.contains(PhotosMarker) || request.markers.contains(ReviewsMarker),
      request.markers.contains(BuyMarker) || request.markers.contains(SearchMarker),
      request.hasHotelMarker,
      request.from == response.from
    ).map(f => if (f) "1" else "0").mkString(" ")

    val relevance = Relevance(request, response)
    if (log.isDebugEnabled) {
      log.debug(s"For $request relevance = [$relevance] factors = [$factors]")
    }

    protobufSerializer.serialize(
      data.toString,
      serpInfo,
      relevance,
      factors,
      humanReadable
    )
  }

}
