package mordabackend

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil/client"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/mordabackend/models"
)

type Config struct {
	BaseURL                 string        `config:"mordabackend-baseurl,required"`
	Timeout                 time.Duration `config:"mordabackend-timeout,required"`
	CircuitBreakerThreshold int           `config:"mordabackend-circuitbreakerthreshold,required"`
	CircuitBreakerTimeout   time.Duration `config:"mordabackend-circuitbreakertimeout,required"`
}

var DefaultConfig = Config{
	Timeout:                 1 * time.Minute,
	CircuitBreakerThreshold: 50,
	CircuitBreakerTimeout:   5 * time.Minute,
}

type MordaBackendClient struct {
	cfg        *Config
	logger     log.Logger
	httpClient *client.HTTPClient
}

func NewMordaBackendClient(cfg *Config, logger log.Logger) (*MordaBackendClient, error) {
	return NewMordaBackendClientWithTransport(cfg, logger, nil)
}

func NewMordaBackendClientWithTransport(
	cfg *Config, logger log.Logger, transport http.RoundTripper,
) (*MordaBackendClient, error) {
	httpClient, err := client.NewHTTPClientWithTransport(cfg.BaseURL, cfg.Timeout, transport,
		cfg.CircuitBreakerThreshold, cfg.CircuitBreakerTimeout, client.ContentTypeJSON, logger, nil)
	if err != nil {
		return nil, err
	}
	return &MordaBackendClient{
		cfg:        cfg,
		logger:     logger,
		httpClient: httpClient,
	}, nil
}

func (c *MordaBackendClient) Search(ctx context.Context, request *models.SearchRequest) (*models.SearchResponse, error) {
	const fnName = "MordaBackendClient.Search"

	var response models.SearchResponse
	err := c.httpClient.Get(ctx, "/ru/search/search/", request, &response)
	if err != nil {
		return nil, err
	}
	if response.Errors != nil && len(response.Errors.(map[string]interface{})) > 0 {
		return nil, client.NewNotRetryableError(fmt.Sprintf("%s: request %v failed: %v", fnName, request, response.Errors))
	}

	return &response, nil
}
