package pathfinderproxy

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil/client"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/pathfinderproxy/models"
)

const (
	clientRequestStateUnknown = iota
	clientRequestStateRequested
	clientRequestStateRequestDone
)

type Config struct {
	BaseURL                 string        `config:"pathfinderproxy-baseurl,required"`
	Timeout                 time.Duration `config:"pathfinderproxy-timeout,required"`
	StateTimeout            time.Duration `config:"pathfinderproxy-statetimeout,required"`
	CircuitBreakerThreshold int           `config:"pathfinderproxy-circuitbreakerthreshold,required"`
	CircuitBreakerTimeout   time.Duration `config:"pathfinderproxy-circuitbreakertimeout,required"`
}

var DefaultConfig = Config{
	Timeout:                 20 * time.Second,
	StateTimeout:            1 * time.Minute,
	CircuitBreakerThreshold: 50,
	CircuitBreakerTimeout:   5 * time.Minute,
}

type PathfinderProxyClient struct {
	cfg          *Config
	logger       log.Logger
	httpClient   *client.HTTPClient
	stateStorage *client.ClientRequestStateStorage
}

func NewPathfinderProxyClient(ctx context.Context, cfg *Config, logger log.Logger) (*PathfinderProxyClient, error) {
	return NewPathfinderProxyClientWithTransport(ctx, cfg, logger, nil)
}

func NewPathfinderProxyClientWithTransport(
	ctx context.Context, cfg *Config, logger log.Logger, transport http.RoundTripper,
) (*PathfinderProxyClient, error) {
	httpClient, err := client.NewHTTPClientWithTransport(cfg.BaseURL, cfg.Timeout, transport,
		cfg.CircuitBreakerThreshold, cfg.CircuitBreakerTimeout, client.ContentTypeJSON, logger, nil)
	if err != nil {
		return nil, err
	}
	stateStorage := client.NewClientStateStorage(clientRequestStateUnknown, cfg.StateTimeout)
	stateStorage.Run(ctx)
	return &PathfinderProxyClient{
		cfg:          cfg,
		logger:       logger,
		httpClient:   httpClient,
		stateStorage: stateStorage,
	}, nil
}

func (c *PathfinderProxyClient) TransfersWithPrices(
	ctx context.Context, request *models.TransferVariantsWithPricesRequest, header http.Header,
) (*models.TransferVariantsWithPricesResponse, error) {
	const fnName = "PathfinderProxyClient.TransfersWithPrices"

	requestKey := request
	requestKey.RID = ""
	requestState := c.stateStorage.Get(requestKey)

	var response models.TransferVariantsWithPricesResponse
	var err error
	if requestState != clientRequestStateRequested {
		err = c.httpClient.GetWithHeader(ctx, "/service/transfers-with-prices/", request, header, &response)
	} else {
		err = c.httpClient.GetWithHeader(ctx, "/service/transfers-with-prices/poll/", request, header, &response)
	}

	if err != nil {
		return nil, err
	}
	switch response.Status {
	case models.QueryStatusDone:
		c.stateStorage.Set(requestKey, clientRequestStateRequestDone)
	case models.QueryStatusQuerying:
		c.stateStorage.Set(requestKey, clientRequestStateRequested)
	default:
		c.stateStorage.Set(requestKey, clientRequestStateUnknown)
		return nil, client.NewNotRetryableError(fmt.Sprintf("%s: request %v done with status=%s", fnName, request, response.Status))
	}

	//TODO: https://st.yandex-team.ru/TRAINS-6065 pathfinder-proxy: убрать отдельный апи для пуллинга

	return &response, nil
}
