package trainapi

import (
	"context"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil/client"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/trainapi/models"
)

type Config struct {
	BaseURL string        `config:"trainapi-baseurl,required"`
	Timeout time.Duration `config:"trainapi-timeout,required"`
}

var DefaultConfig = Config{
	Timeout: 20 * time.Second,
}

type TrainAPIClient struct {
	cfg        *Config
	logger     log.Logger
	httpClient *client.HTTPClient
}

func NewTrainAPIClient(cfg *Config, logger log.Logger) (*TrainAPIClient, error) {
	return NewTrainAPIClientWithTransport(cfg, logger, nil)
}

func NewTrainAPIClientWithTransport(
	cfg *Config, logger log.Logger, transport http.RoundTripper,
) (*TrainAPIClient, error) {
	httpClient, err := client.NewHTTPClientWithTransport(cfg.BaseURL, cfg.Timeout, transport,
		0, 0, client.ContentTypeJSON, logger, nil)
	if err != nil {
		return nil, err
	}
	return &TrainAPIClient{
		cfg:        cfg,
		logger:     logger,
		httpClient: httpClient,
	}, nil
}

func (c *TrainAPIClient) ActivePartners(ctx context.Context, request *models.ActivePartnersRequest) (*models.ActivePartnersResponse, error) {
	var response models.ActivePartnersResponse
	err := c.httpClient.Get(ctx, "/ru/api/active-partners/", request, &response)
	return &response, err
}

func (c *TrainAPIClient) TrainTariffs(ctx context.Context, request *models.TrainTariffsRequest, header http.Header) (*models.TrainTariffsResponse, error) {
	var response models.TrainTariffsResponse
	err := c.httpClient.GetWithHeader(ctx, "/ru/api/segments/train-tariffs/", request, header, &response)

	if err != nil {
		return nil, err
	}

	return &response, nil
}
