package models

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/protobuf/types/known/timestamppb"

	travel "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"

	api "a.yandex-team.ru/travel/trains/search_api/api/tariffs"
)

var threadTitleTypeByName = map[string]rasp.TThreadTitle_EType{
	"default":  rasp.TThreadTitle_TYPE_DEFAULT,
	"suburban": rasp.TThreadTitle_TYPE_SUBURBAN,
	"mta":      rasp.TThreadTitle_TYPE_MTA,
}

var transportTypeByName = map[string]rasp.TTransport_EType{
	"unknown":    rasp.TTransport_TYPE_UNKNOWN,
	"train":      rasp.TTransport_TYPE_TRAIN,
	"plane":      rasp.TTransport_TYPE_PLANE,
	"bus":        rasp.TTransport_TYPE_BUS,
	"suburban":   rasp.TTransport_TYPE_SUBURBAN,
	"helicopter": rasp.TTransport_TYPE_HELICOPTER,
	"water":      rasp.TTransport_TYPE_WATER,
}

var currencyByName = map[string]travel.ECurrency{
	"RUB": travel.ECurrency_C_RUB,
	"USD": travel.ECurrency_C_USD,
	"EUR": travel.ECurrency_C_EUR,
	"JPY": travel.ECurrency_C_JPY,
}

func MapDirectionTariffTrainJSONToAPI(ctx context.Context, modelTrains ...*DirectionTariffTrain) ([]*api.DirectionTariffTrain, error) {
	const funcName = "trains.internal.pkg.tariffs.model.MapDirectionTariffTrainJSONToAPI"

	span, _ := opentracing.StartSpanFromContext(ctx, funcName)
	defer span.Finish()

	result := make([]*api.DirectionTariffTrain, len(modelTrains))
	for i, modelTrain := range modelTrains {
		arrival := timestamppb.New(modelTrain.Arrival.UTC())
		departure := timestamppb.New(modelTrain.Departure.UTC())

		var brokenClasses *api.TariffBrokenClasses
		if modelTrain.BrokenClasses != nil {
			brokenClasses = &api.TariffBrokenClasses{
				Unknown:     modelTrain.BrokenClasses.Unknown,
				Soft:        modelTrain.BrokenClasses.Soft,
				Platzkarte:  modelTrain.BrokenClasses.Platzkarte,
				Compartment: modelTrain.BrokenClasses.Compartment,
				Suite:       modelTrain.BrokenClasses.Suite,
				Common:      modelTrain.BrokenClasses.Common,
				Sitting:     modelTrain.BrokenClasses.Sitting,
			}
		}

		var titleDict *rasp.TThreadTitle
		if modelTrain.TitleDict != nil {
			titleDict = &rasp.TThreadTitle{
				Type:          threadTitleTypeByName[modelTrain.TitleDict.Type],
				TransportType: parseTransportType(modelTrain.TitleDict.TransportType),
				TitleParts:    make([]*rasp.TThreadTitlePart, len(modelTrain.TitleDict.TitleParts)),
				IsCombined:    modelTrain.TitleDict.IsCombined,
				IsRing:        modelTrain.TitleDict.IsRing,
			}
			for i, p := range modelTrain.TitleDict.TitleParts {
				titlePart, err := parseTitlePart(p)
				if err != nil {
					return nil, fmt.Errorf(
						"%s: parsing title part '%s': %w", funcName, p, err,
					)
				}
				titleDict.TitleParts[i] = titlePart
			}
		}

		train := &api.DirectionTariffTrain{
			Arrival:            arrival,
			ArrivalStationId:   modelTrain.ArrivalStationID,
			Departure:          departure,
			DepartureStationId: modelTrain.DepartureStationID,
			Number:             modelTrain.Number,
			DisplayNumber:      modelTrain.DisplayNumber,
			HasDynamicPricing:  modelTrain.HasDynamicPricing,
			TwoStorey:          modelTrain.TwoStorey,
			IsSuburban:         modelTrain.IsSuburban,
			CoachOwners:        modelTrain.CoachOwners,
			ElectronicTicket:   modelTrain.ElectronicTicket,
			FirstCountryCode:   modelTrain.FirstCountryCode,
			LastCountryCode:    modelTrain.LastCountryCode,
			Places:             make([]*api.TrainPlace, len(modelTrain.Places)),
			BrokenClasses:      brokenClasses,
			TitleDict:          titleDict,
			Provider:           modelTrain.Provider,
			RawTrainName:       modelTrain.RawTrainName,
			HasThread:          modelTrain.HasThread,
		}

		for i, p := range modelTrain.Places {
			place, err := parseTrainPlace(p)
			if err != nil {
				return nil, fmt.Errorf(
					"%s: parsing train place '%v': %w", funcName, p, err,
				)
			}
			train.Places[i] = place
		}

		result[i] = train
	}
	return result, nil
}

func parseTransportType(modelTransportType string) rasp.TTransport_EType {
	transportType, wellKnown := transportTypeByName[modelTransportType]
	if wellKnown {
		return transportType
	} else {
		return rasp.TTransport_TYPE_UNKNOWN
	}
}

func parseTitlePart(modelTitlePart string) (*rasp.TThreadTitlePart, error) {
	if len(modelTitlePart) < 2 {
		return nil, fmt.Errorf("length of point key is too small")
	}

	id, err := strconv.ParseUint(modelTitlePart[1:], 10, 32)
	if err != nil {
		return nil, fmt.Errorf("incorrect point format")
	}

	if strings.HasPrefix(modelTitlePart, "s") {
		return &rasp.TThreadTitlePart{
			StationId: uint32(id),
		}, nil
	} else if strings.HasPrefix(modelTitlePart, "c") {
		return &rasp.TThreadTitlePart{
			SettlementId: uint32(id),
		}, nil
	} else {
		return nil, fmt.Errorf("unknown point type")
	}
}

func parseCurrency(modelCurrency string) travel.ECurrency {
	currency, wellKnown := currencyByName[modelCurrency]
	if wellKnown {
		return currency
	} else {
		return travel.ECurrency_C_UNKNOWN
	}
}

func parsePrice(modelPrice string, currency travel.ECurrency) (*travel.TPrice, error) {
	price := travel.TPrice{
		Currency:  currency,
		Precision: 0,
	}

	pointPos := strings.LastIndex(modelPrice, ".")
	if pointPos != -1 {
		price.Precision = int32(len(modelPrice) - pointPos - 1)
		modelPrice = modelPrice[:pointPos] + modelPrice[pointPos+1:]
	}

	amount, err := strconv.ParseInt(modelPrice, 10, 64)
	if err != nil {
		return nil, fmt.Errorf("parsing int '%s' : %w", modelPrice, err)
	}
	price.Amount = amount

	return &price, nil
}

func parseTrainPlace(modelPlace *TrainPlace) (*api.TrainPlace, error) {
	currency := travel.ECurrency_C_UNKNOWN

	if modelPlace.Price != nil {
		currency = parseCurrency(modelPlace.Price.Currency)
	}

	price, err := parsePrice(modelPlace.Price.Value, currency)
	if err != nil {
		return nil, fmt.Errorf("parsing price '%s' : %w", modelPlace.Price.Value, err)
	}

	var priceDetails *api.TrainPlacePriceDetails
	if modelPlace.PriceDetails != nil {
		ticketPrice, err := parsePrice(modelPlace.PriceDetails.TicketPrice, currency)
		if err != nil {
			return nil, fmt.Errorf("parsing ticket price '%s' : %w", modelPlace.PriceDetails.TicketPrice, err)
		}

		servicePrice, err := parsePrice(modelPlace.PriceDetails.ServicePrice, currency)
		if err != nil {
			return nil, fmt.Errorf("parsing service price '%s' : %w", modelPlace.PriceDetails.ServicePrice, err)
		}

		fee, err := parsePrice(modelPlace.PriceDetails.Fee, currency)
		if err != nil {
			return nil, fmt.Errorf("parsing fee '%s' : %w", modelPlace.PriceDetails.Fee, err)
		}

		priceDetails = &api.TrainPlacePriceDetails{
			TicketPrice:   ticketPrice,
			ServicePrice:  servicePrice,
			Fee:           fee,
			SeveralPrices: modelPlace.PriceDetails.SeveralPrices,
		}
	}

	return &api.TrainPlace{
		CoachType:              modelPlace.CoachType,
		ServiceClass:           modelPlace.ServiceClass,
		Count:                  modelPlace.Count,
		LowerCount:             modelPlace.LowerCount,
		UpperCount:             modelPlace.UpperCount,
		LowerSideCount:         modelPlace.LowerSideCount,
		UpperSideCount:         modelPlace.UpperSideCount,
		MaxSeatsInTheSameCar:   modelPlace.MaxSeatsInTheSameCar,
		HasNonRefundableTariff: modelPlace.HasNonRefundableTariff,
		Price:                  price,
		PriceDetails:           priceDetails,
	}, nil
}
