package main

import (
	"context"
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	coremetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcmetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	gcodes "google.golang.org/grpc/codes"
	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	pbapi "a.yandex-team.ru/travel/trains/search_api/api"
	"a.yandex-team.ru/travel/trains/search_api/internal/app"
	"a.yandex-team.ru/travel/trains/search_api/internal/handler"
)

const (
	serviceName            = "trains.search_api"
	grpcGatewayServiceName = "search_api_service"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()

	cfgLoader := configuration.NewDefaultConfitaLoader()
	err := cfgLoader.Load(ctx, &Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		return
	}

	logger, err := logging.NewDeploy(&Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(Cfg.Tvm.WhiteList)
	if Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient()
		if err != nil {
			fmt.Println("failed to create tvm client:", err)
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTags()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, Cfg.Metrics.PerfMetricsRefreshInterval)

	unifiedAgentClient, err := unifiedagent.NewGrpcClient(&Cfg.UnifiedAgentClient, logger, nil)
	if err != nil {
		logger.Fatal("Error while starting UnifiedAgent client: %s", log.Error(err))
	}

	app, err := app.NewApp(logger, &Cfg.App)
	if err != nil {
		logger.Fatal("Error while running application", log.Error(err))
	}
	defer app.Destroy()

	httpHandler := handler.NewHTTPHandler(app, logger)
	grpcHandler := handler.NewGRPCHandler(app, unifiedAgentClient, logger)

	httpRouteBuilders := []httpserver.RouteBuilder{
		httpHandler.GetRouteBuilder(),
	}

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		grpcHandler.GetServiceRegisterer(),
	}

	healthServer := ghealth.NewServer()
	// The health-checking logic of a specific application should be implemented below
	go func() {
		for range time.Tick(2 * time.Second) {
			healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
		}
	}()

	grpcGateway := grpcgateway.NewGateway(
		&Cfg.GrpcGateway,
		grpcgateway.NewService(grpcGatewayServiceName, "", Cfg.Grpc.Addr, pbapi.RegisterSearchApiServiceV1HandlerFromEndpoint, nil))

	// running server
	wg := sync.WaitGroup{}
	wg.Add(4)

	go func() {
		defer wg.Done()
		server := httpserver.NewHTTPServerBuilder(Cfg.HTTP, logger, httpRouteBuilders, rootRegistry.WithPrefix("http")).
			WithTVM(tvmClient, tvmAllowedIds).
			WithMetricsMiddlewareOptions(
				httpmetrics.WithDurationBuckets(
					coremetrics.MakeExponentialDurationBuckets(5*time.Millisecond, 1.5, 20))).
			Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		server := grpcserver.NewGrpcServerBuilder(Cfg.Grpc, logger).
			WithMetrics(
				rootRegistry.WithPrefix("grpc"),
				grpcmetrics.WithCodesCount(gcodes.InvalidArgument, gcodes.Unavailable)).
			WithRegisterers(grpcServiceRegisterers...).
			WithOptionalTVM(tvmClient, tvmAllowedIds).
			WithHealthServer(healthServer).
			Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		if !Cfg.GrpcGateway.Enabled {
			return
		}
		err = grpcGateway.Run(ctx)
		if err != nil {
			logger.Info("GRPC-gateway server closed", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
