package app

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	"a.yandex-team.ru/travel/trains/library/go/grpcutil/clients/trainbanditapi"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/ytlock"

	tariffmodels "a.yandex-team.ru/travel/trains/library/go/tariffs/models"
	pcpb "a.yandex-team.ru/travel/trains/search_api/api/price_calendar"
	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	api "a.yandex-team.ru/travel/trains/search_api/api/tariffs"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/filters"
	dirmodels "a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/query"
	"a.yandex-team.ru/travel/trains/search_api/internal/indexer"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/express"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/feeservice"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/geo"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	pkglogbroker "a.yandex-team.ru/travel/trains/search_api/internal/pkg/logbroker"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/pricecalendar"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/regioncapital"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
	tariffcache "a.yandex-team.ru/travel/trains/search_api/internal/pkg/tariffs/cache"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/templater"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/traincity"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/updater"
	"a.yandex-team.ru/travel/trains/search_api/internal/searcher"
	"a.yandex-team.ru/travel/trains/search_api/internal/searcher/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo"
)

type App struct {
	// common
	logger  log.Logger
	cfg     *Config
	cancels []func()

	ytLockClient        yt.Client
	geobaseClient       geo.Geobase
	cacheUpdater        *updater.Updater
	repositoriesUpdater *updater.Updater

	// components
	searcher          *searcher.Searcher
	directionProvider *direction.Provider
	pointParser       *points.Parser
	indexer           *indexer.Indexer
	feeService        *feeservice.FeeService

	// tariffs
	tariffCache    *tariffcache.TariffCache
	tariffConsumer *logbroker.Consumer
	tariffProducer *logbroker.Producer

	// repos
	repoRegistry        *registry.RepositoryRegistry
	regionCapitalRepo   *regioncapital.Repository
	scheduleRepository  *schedule.Repository
	trainCityRepository *traincity.Repository
	expressRepository   *express.Repository

	// i18n
	keyset                i18n.FileKeyset
	trainTitleTranslator  *i18n.TrainTitleTranslator
	timeTranslator        *i18n.TimeTranslator
	translatableFactory   *i18n.TranslatableFactory
	linguisticsTranslator *i18n.LinguisticsTranslator
	durationTranslator    *i18n.DurationTranslator
	carTypeTranslator     *i18n.CarTypeTranslator

	// seo
	seoUpdater *updater.Updater
	seoService *seo.Service

	// price calendar
	PriceCalendarService *pricecalendar.TariffCacheBasedService
}

func NewApp(logger log.Logger, cfg *Config) (*App, error) {
	const funcName = "app.NewApp"
	ctx, cancelCtx := context.WithCancel(context.Background())
	a := &App{
		logger:  logger,
		cfg:     cfg,
		cancels: []func(){cancelCtx},
	}

	var err error
	defer func() {
		if err != nil {
			a.Destroy()
		}
	}()

	a.geobaseClient, err = geo.NewGeobase(&cfg.Geobase, logger)
	if err != nil {
		return nil, xerrors.Errorf("%s: NewGeobase fails: %w", funcName, err)
	}
	a.cancels = append(a.cancels, a.geobaseClient.Destroy)

	a.ytLockClient, err = ythttp.NewClient(
		&yt.Config{
			Proxy: a.cfg.YtLockCluster,
			Token: a.cfg.YtToken,
		},
	)
	if err != nil {
		return nil, xerrors.Errorf("%s: failed to create ytLock client: %w", funcName, err)
	}

	err = a.buildI18n()
	if err != nil {
		return nil, xerrors.Errorf("%s: i18n build failed: %w", funcName, err)
	}

	err = a.buildTariffProviders()
	if err != nil {
		return nil, xerrors.Errorf("%s: tariff providers build failed: %w", funcName, err)
	}

	err = a.buildRepositories()
	if err != nil {
		return nil, xerrors.Errorf("%s: repositories build failed: %w", funcName, err)
	}

	err = a.buildFeeService(ctx)
	if err != nil {
		return nil, xerrors.Errorf("%s: fee service build failed: %w", funcName, err)
	}

	a.directionProvider = direction.NewProvider(
		&a.cfg.Direction,
		a.logger,
		a.pointParser,
		a.repoRegistry,
		a.scheduleRepository,
		a.tariffCache,
		filters.NewFactory(
			a.logger,
			a.translatableFactory,
			filters.WithConfigsLoading(a.logger, &a.cfg.Direction.Filters),
		),
		a.expressRepository,
		a.trainTitleTranslator,
		a.timeTranslator,
		a.translatableFactory,
	)

	a.PriceCalendarService = pricecalendar.NewService(
		a.pointParser,
		a.tariffCache,
		a.repoRegistry,
		a.expressRepository,
		a.scheduleRepository,
		a.feeService,
		a.logger,
	)

	a.searcher, err = searcher.NewSearcher(ctx, logger, a.PriceCalendarService, cfg.Searcher)
	if err != nil {
		return nil, xerrors.Errorf("%s: searcher creation fails: %w", funcName, err)
	}

	a.indexer = indexer.New(logger, a.tariffProducer)

	err = a.buildSeo()
	if err != nil {
		return nil, xerrors.Errorf("%s: seo build failed: %w", funcName, err)
	}

	return a.run(ctx)
}

func (a *App) Destroy() {
	for _, cancel := range a.cancels {
		cancel()
	}
}

func (a *App) Search(
	ctx context.Context,
	pointFrom string, pointTo string,
	when string, returnWhen string,
	pinForwardSegmentID string, pinBackwardSegmentID string,
	onlyDirect bool, onlyOwnedPrices bool,
	userArgs models.UserArgs,
	testContext models.TestContext,
) (*models.TransferVariantsResponse, error) {
	return a.searcher.Search(
		ctx,
		pointFrom, pointTo,
		when, returnWhen,
		pinForwardSegmentID, pinBackwardSegmentID,
		onlyDirect, onlyOwnedPrices,
		userArgs, testContext,
	)
}

func (a *App) Direction(
	ctx context.Context, rawQuery *query.RawDirectionQuery,
) (dirmodels.Response, error) {
	return a.directionProvider.GetDirections(ctx, rawQuery)
}

func (a *App) OpenDirection(
	ctx context.Context, rawQuery *query.RawDirectionQuery,
) (dirmodels.Response, error) {
	return a.directionProvider.GetOpenDirections(ctx, rawQuery)
}

func (a *App) Index(
	ctx context.Context,
	departurePointExpressID int,
	arrivalPointExpressID int,
	departureDate time.Time,
	info []*tariffmodels.DirectionTariffTrain,
) error {
	return a.indexer.Index(ctx, departurePointExpressID, arrivalPointExpressID, departureDate, info)
}

func (a *App) SeoDirection(ctx context.Context, fromSlug, toSlug string, language lang.Lang) (*seopb.SeoDirectionResponse, error) {
	return a.seoService.SeoDirection(ctx, fromSlug, toSlug, language)
}

func (a *App) PriceCalendar(ctx context.Context, pointFrom string, pointTo string, userArgs models.UserArgs) (*pcpb.PriceCalendarResponse, error) {
	return a.PriceCalendarService.PriceCalendar(ctx, pointFrom, pointTo, userArgs)
}

func (a *App) buildSeo() error {
	templater.SetTranslator(a.translatableFactory)
	seoResourceReader := resourcestorage.NewS3StorageReader(
		resourcestorage.S3StorageConfig(a.cfg.Seo.S3Storage),
		a.cfg.Seo.S3AccessKey,
		a.cfg.Seo.S3Secret,
	)
	seoKeySets := seo.NewResource(a.cfg.Seo.SnapshotDirectory, a.logger, seoResourceReader)
	a.seoService = seo.NewService(
		a.durationTranslator,
		a.translatableFactory,
		a.carTypeTranslator,
		a.tariffCache,
		a.repoRegistry,
		a.expressRepository,
		a.scheduleRepository,
		a.pointParser,
		seoKeySets,
		a.logger,
	)
	seoKeySets.SetResourceUpdatedCallback(func() error {
		err := a.seoService.UpdateKeySets()
		if err != nil {
			return err
		}
		commonKs := seoKeySets.GetKeySet(seo.KeySetNameCommon)
		a.durationTranslator.UpdateKeySet(commonKs)
		a.carTypeTranslator.UpdateKeySet(commonKs)
		return nil
	})
	a.seoUpdater = updater.NewUpdater("seo", a.logger)
	a.seoUpdater.AddUpdateRule("keysets", 5*time.Minute, seoKeySets.UpdateResource)
	return nil
}

func (a *App) buildI18n() error {
	const funcName = "trains.search_api.internal.pkg.app.App.buildI18n"

	var err error
	a.keyset, err = i18n.ReadKeyset(a.cfg.I18n.KeysetPath)
	if err != nil {
		return fmt.Errorf("%s: keyset creation fails: %w", funcName, err)
	}
	a.trainTitleTranslator = i18n.NewTrainTitleTranslator(a.keyset)
	a.timeTranslator = i18n.NewTimeTranslator(i18n.WithKeysetPath(a.logger, &a.cfg.I18n))
	a.linguisticsTranslator = i18n.NewLinguisticsTranslator(
		a.geobaseClient, a.keyset, a.cfg.Direction.LanguageFallbacks, a.cfg.Direction.LanguageCaseFallbacks)
	a.translatableFactory = i18n.NewTranslatableFactory(a.linguisticsTranslator, a.keyset)
	a.durationTranslator = i18n.NewDurationTranslator(nil)
	a.carTypeTranslator = i18n.NewCarTypeTranslator(nil, a.linguisticsTranslator)
	return nil
}

func (a *App) buildTariffProviders() error {
	const funcName = "trains.search_api.internal.pkg.app.App.buildTariffProviders"
	var err error

	a.tariffCache = tariffcache.NewTariffCache(&a.cfg.TariffCache, a.logger)

	if !a.cfg.TariffCache.Enable {
		a.logger.Info("tariff cache disabled")
		return nil
	}

	snapshotsResourceReader := resourcestorage.NewS3StorageReader(
		resourcestorage.S3StorageConfig(a.cfg.TariffCache.S3Storage),
		a.cfg.TariffCache.S3AccessKey,
		a.cfg.TariffCache.S3Secret,
	)

	tariffCacheSnapshotTS := time.Now().Add(-a.cfg.TariffCache.SnapshotTTL)
	if !a.cfg.TariffCache.SkipTariffSnapshotLoading {
		tariffCacheSnapshotTS, err = snapshotsResourceReader.GetTimestamp(a.cfg.TariffCache.SnapshotDirectory)
		if err != nil {
			return xerrors.Errorf("%s: failed to load tariff cache snapshot: %w", funcName, err)
		}

		// for logbroker lag recovery
		tariffCacheSnapshotTS = tariffCacheSnapshotTS.Add(-time.Hour)

		tariffCacheLoader := resourcestorage.NewLoader(&api.DirectionTariffInfo{}, a.cfg.TariffCache.SnapshotDirectory, snapshotsResourceReader, a.logger)
		n, err := tariffCacheLoader.Load(a.tariffCache)
		if err != nil {
			a.logger.Errorf("Can not load snapshot for %s: %s", a.cfg.TariffCache.SnapshotDirectory, err.Error())
		} else {
			a.logger.Infof("Loaded %d records for %s", n, a.cfg.TariffCache.SnapshotDirectory)
		}
	}
	a.logger.Info("tariff cache timestamp", log.Time("timestamp", tariffCacheSnapshotTS))

	a.tariffConsumer, err = pkglogbroker.NewConsumer(&a.cfg.LogbrokerConsumer, tariffCacheSnapshotTS, a.ytLockClient, a.logger)
	if err != nil {
		return xerrors.Errorf("%s: failed to create tariff consumer: %w", funcName, err)
	}
	a.tariffProducer, err = pkglogbroker.NewProducer(&a.cfg.LogbrokerProducer, a.logger)
	if err != nil {
		return xerrors.Errorf("%s: failed to create tariff producer: %w", funcName, err)
	}
	return nil
}

func (a *App) buildFeeService(ctx context.Context) error {
	const funcName = "trains.search_api.internal.pkg.app.App.buildFeeService"
	banditClient, err := trainbanditapi.CreateClient(ctx, a.cfg.TrainBanditAPI)
	if err != nil {
		return xerrors.Errorf("%s: failed to create bandit client: %w", funcName, err)
	}
	a.feeService = feeservice.NewService(banditClient, a.logger)
	return nil
}

func (a *App) buildRepositories() error {
	a.repoRegistry = registry.NewRepositoryRegistry(a.logger)

	dictsResourceReader := resourcestorage.NewS3StorageReader(
		resourcestorage.S3StorageConfig(a.cfg.Dict.S3Storage),
		a.cfg.Dict.S3AccessKey,
		a.cfg.Dict.S3Secret,
	)
	a.repositoriesUpdater = registry.NewRepositoryUpdater(&a.cfg.Dict, a.logger, dictsResourceReader, a.repoRegistry)

	a.regionCapitalRepo = regioncapital.NewRepository(a.logger, a.geobaseClient, a.repoRegistry)
	a.pointParser = points.NewParser(a.repoRegistry, a.regionCapitalRepo)

	a.scheduleRepository = schedule.NewRepository(a.logger, a.repoRegistry)

	a.trainCityRepository = traincity.NewRepository(a.logger, a.repoRegistry)
	a.expressRepository = express.NewRepository(a.logger, a.repoRegistry, a.trainCityRepository)

	a.cacheUpdater = updater.NewUpdater("cache", a.logger)
	a.cacheUpdater.AddUpdateRule("schedule", 5*time.Minute, a.scheduleRepository.UpdateCache)
	a.cacheUpdater.AddUpdateRule("traincity", 5*time.Minute, a.trainCityRepository.UpdateCache)
	a.cacheUpdater.AddUpdateRule("express", 5*time.Minute, a.expressRepository.UpdateCache)

	return nil
}

func (a *App) run(ctx context.Context) (*App, error) {
	const funcName = "trains.search_api.internal.pkg.app.App.run"

	a.searcher.Run(ctx)

	if err := a.repositoriesUpdater.UpdateAll(ctx); err != nil {
		return nil, xerrors.Errorf("%s: failed to update dicts: %w", funcName, err)
	}
	a.repositoriesUpdater.RunUpdating(ctx)

	if err := a.cacheUpdater.UpdateAll(ctx); err != nil {
		return nil, xerrors.Errorf("%s: failed to update caches: %w", funcName, err)
	}
	a.cacheUpdater.RunUpdating(ctx)

	if err := a.seoUpdater.UpdateAll(ctx); err != nil {
		return nil, xerrors.Errorf("%s: failed to update seo: %w", funcName, err)
	}
	a.seoUpdater.RunUpdating(ctx)

	if a.cfg.TariffCache.Enable {
		if err := a.tariffConsumer.Run(ctx); err != nil {
			return nil, xerrors.Errorf("%s: failed to run tariffConsumer: %w", funcName, err)
		}
		if err := a.tariffProducer.Run(ctx); err != nil {
			return nil, xerrors.Errorf("%s: failed to run tariffProducer: %w", funcName, err)
		}
		go tariffcache.RunConsuming(ctx, a.tariffCache, a.tariffConsumer, a.logger)

		a.tariffCache.Run(ctx)
		a.runTariffSnapshotDumping(ctx)
	}
	return a, nil
}

func (a *App) runTariffSnapshotDumping(ctx context.Context) {
	var isSnapshotDumpMaster bool
	defer func() {
		a.logger.Info("choosing snapshot dumping master",
			log.Bool("is_master", isSnapshotDumpMaster),
			log.Duration("dump_period", a.cfg.TariffCache.SnapshotDumpPeriod),
		)
	}()

	if a.cfg.TariffCache.SnapshotDumpLockPath == "" || a.cfg.TariffCache.SkipTariffSnapshotLoading {
		return
	}

	snapshotsResourceWriter := resourcestorage.NewS3StorageWriter(
		resourcestorage.S3StorageConfig(a.cfg.TariffCache.S3Storage),
		a.cfg.TariffCache.S3AccessKey,
		a.cfg.TariffCache.S3Secret,
	)

	lockPath := ypath.Path(a.cfg.TariffCache.SnapshotDumpLockPath)
	tariffCacheSnapshotLock := ytlock.NewLock(a.ytLockClient, lockPath)
	if _, err := tariffCacheSnapshotLock.Acquire(ctx); err != nil {
		return
	}

	isSnapshotDumpMaster = true
	dumper := resourcestorage.NewDumper(
		a.tariffCache,
		a.cfg.TariffCache.SnapshotDirectory,
		snapshotsResourceWriter,
		2,
		a.logger,
	)
	dumper.RunPeriodic(a.cfg.TariffCache.SnapshotDumpPeriod, ctx)
}
