package eventdate

import (
	"time"

	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/clock"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
)

type Query interface {
	GetMinDate() time.Time
	Filter(segments schedule.Segments) schedule.Segments
}

func BuildFixedDateQuery(localDate time.Time, loc *time.Location) *FixedDateQuery {
	minDate := clock.Now().In(loc)
	if !date.DateFromTime(minDate).Equal(date.DateFromTime(localDate)) {
		minDate = date.ReplaceLocation(date.DateFromTime(localDate), loc)
	}
	maxDate := date.ReplaceLocation(date.DateFromTime(localDate.Add(units.Day)), loc)
	return &FixedDateQuery{
		minDate: minDate,
		maxDate: maxDate,
	}
}

type FixedDateQuery struct {
	minDate time.Time
	maxDate time.Time
}

func (d *FixedDateQuery) GetMinDate() time.Time {
	return d.minDate
}

func (d *FixedDateQuery) Filter(segments schedule.Segments) schedule.Segments {
	filtered := make(schedule.Segments, 0)
	for _, s := range segments {
		if s.DepartureDateTime.Before(d.maxDate) {
			filtered = append(filtered, s)
		}
	}
	return filtered
}

func BuildFirstFutureDateQuery(loc *time.Location) *FirstFutureDateQuery {
	tomorrowUTC := date.DateFromTime(clock.Now().In(loc).Add(units.Day))
	tomorrow := date.ReplaceLocation(tomorrowUTC, loc)
	return &FirstFutureDateQuery{
		minDate: tomorrow,
	}
}

type FirstFutureDateQuery struct {
	minDate time.Time
}

func (d *FirstFutureDateQuery) GetMinDate() time.Time {
	return d.minDate
}

func (d *FirstFutureDateQuery) Filter(segments schedule.Segments) schedule.Segments {
	_, filtered := schedule.FindFirstDateSegments(segments)
	return filtered
}
