package filters

import (
	"sort"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/serialization"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type brandIDOption int

func (b brandIDOption) isVariantOption() {}

type brandDumpParams struct {
	id          int
	isHighSpeed bool
	title       string
}

type brandFilterKey struct {
	id              int
	normalizedTitle string
}

func (p *brandDumpParams) makeBrandFilterKey() *brandFilterKey {
	return &brandFilterKey{
		id:              p.id,
		normalizedTitle: helpers.Normalize(p.title),
	}
}

type BrandFilter struct {
	BaseFilter
	selectedIDs        map[int]bool
	selectedTitles     map[string]bool
	brandDumpParamsMap map[brandIDOption]*brandDumpParams

	logger                         log.Logger
	language                       lang.Lang
	translatableFactory            *i18n.TranslatableFactory
	deluxeTrainNameFormsToID       map[string]int
	deluxeTrainNamePluralFormsToID map[int]string
}

func NewBrandFilter(
	logger log.Logger,
	language lang.Lang,
	translatableFactory *i18n.TranslatableFactory,
	deluxeTrainNameFormsToID map[string]int,
	deluxeTrainNamePluralFormsToID map[int]string,
) *BrandFilter {
	return &BrandFilter{
		BaseFilter:         NewBaseFilter(brandName),
		selectedIDs:        make(map[int]bool),
		selectedTitles:     make(map[string]bool),
		brandDumpParamsMap: make(map[brandIDOption]*brandDumpParams),

		logger:                         logger,
		language:                       language,
		translatableFactory:            translatableFactory,
		deluxeTrainNameFormsToID:       deluxeTrainNameFormsToID,
		deluxeTrainNamePluralFormsToID: deluxeTrainNamePluralFormsToID,
	}
}

func (f *BrandFilter) LoadSelected(selected []string) error {
	for _, value := range selected {
		if brandID, err := strconv.Atoi(value); err == nil {
			f.selectedIDs[brandID] = true
		} else {
			value = helpers.Normalize(value)
			if brandID, found := f.deluxeTrainNameFormsToID[value]; found {
				f.selectedIDs[brandID] = true
			} else {
				f.selectedTitles[value] = true
			}
		}
	}
	return nil
}

func (f *BrandFilter) BindVariants(variants ...segments.TrainVariant) {
	for _, v := range variants {
		if brand := v.Segment.TrainBrand; brand != nil {
			f.registerBrand(brand)
			f.variantStorage.AddVariant(v, brandIDOption(brand.Id))
		}
	}
}

func (f *BrandFilter) Dump() interface{} {
	var rawIDs []int
	for id := range f.brandDumpParamsMap {
		rawIDs = append(rawIDs, int(id))
	}
	sort.Ints(rawIDs)

	result := make([]models.BrandFilterResponse, 0)
	for _, rawID := range rawIDs {
		id := brandIDOption(rawID)
		dumpParams := f.brandDumpParamsMap[id]
		availableVariants := f.variantStorage.GetAvailableVariantsByOption(id)
		result = append(result, models.BrandFilterResponse{
			Value:        rawID,
			Available:    len(availableVariants) > 0,
			Selected:     f.variantStorage.IsSelectedOption(id),
			MinimumPrice: serialization.DumpPrice(getMinPrice(availableVariants)),
			Title:        dumpParams.title,
			IsHighSpeed:  dumpParams.isHighSpeed,
		})
	}
	return result
}

func (f *BrandFilter) GetSearchParams() map[string][]string {
	const searchParamKey = "highSpeedTrain"

	params := make(map[string][]string)
	for id := range f.brandDumpParamsMap {
		if f.variantStorage.IsSelectedOption(id) {
			params[searchParamKey] = append(params[searchParamKey], strconv.Itoa(int(id)))
		}
	}
	return params
}

func (f *BrandFilter) GetBrandTitle() string {
	var selectedCount int
	var title string

	for id := range f.brandDumpParamsMap {
		if f.variantStorage.IsSelectedOption(id) {
			selectedCount++
			title = f.deluxeTrainNamePluralFormsToID[int(id)]
		}
	}
	if selectedCount == 1 {
		return title
	}
	return ""
}

func (f *BrandFilter) registerBrand(brand *rasp.TNamedTrain) {
	id := brandIDOption(brand.Id)
	params, found := f.brandDumpParamsMap[id]
	if !found {
		f.createBrandDumpParams(brand)
		params = f.brandDumpParamsMap[id]
	}
	if f.isSelectedKey(params.makeBrandFilterKey()) {
		f.variantStorage.SelectOption(id)
	}
}

func (f *BrandFilter) createBrandDumpParams(brand *rasp.TNamedTrain) {
	id := brand.Id
	title := brand.TitleDefault

	namedBrand := f.translatableFactory.TranslatableNamedTrain(brand)
	if translatedTitle, err := namedBrand.Title(f.language, lang.Nominative); err != nil {
		f.logger.Warn("not found brand translation",
			log.Int32("brand_id", id),
			log.String("lang", f.language.String()),
			log.Error(err),
		)
	} else {
		title = translatedTitle
	}

	dumpParams := &brandDumpParams{
		id:          int(id),
		title:       title,
		isHighSpeed: brand.IsHighSpeed,
	}
	f.brandDumpParamsMap[brandIDOption(id)] = dumpParams
}

func (f *BrandFilter) isSelectedKey(key *brandFilterKey) bool {
	return f.selectedIDs[key.id] || f.selectedTitles[key.normalizedTitle]
}
