package filters

import (
	"sort"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/serialization"
)

var InvalidCoachFilterError = xerrors.New("invalid coach filter value")

type coachType int

func (b coachType) isVariantOption() {}

const (
	Unknown coachType = iota
	Soft
	Platzkarte
	Compartment
	Suite
	Common
	Sitting
)

var coachTypeToIDMap = map[string]coachType{
	"unknown":     Unknown,
	"soft":        Soft,
	"platzkarte":  Platzkarte,
	"compartment": Compartment,
	"suite":       Suite,
	"common":      Common,
	"sitting":     Sitting,
}

type CoachTypeFilter struct {
	BaseFilter
}

func NewCoachTypeFilter() *CoachTypeFilter {
	return &CoachTypeFilter{
		BaseFilter: NewBaseFilter(coachTypeName),
	}
}

func (f *CoachTypeFilter) LoadSelected(selected []string) error {
	for _, value := range selected {
		if coachTypeID, found := coachTypeToIDMap[value]; !found {
			return coachTypeLoadSelectedCaller.WrapError(InvalidCoachFilterError)
		} else {
			f.variantStorage.SelectOption(coachTypeID)
		}
	}
	return nil
}

func (f *CoachTypeFilter) BindVariants(variants ...segments.TrainVariant) {
	for _, v := range variants {
		placeGroup := v.Place
		if placeGroup == nil {
			continue
		}
		if id, found := coachTypeToIDMap[placeGroup.CoachType]; found {
			f.variantStorage.AddVariant(v, id)
		}
	}
}

func (f *CoachTypeFilter) Dump() interface{} {
	var values []string
	for value := range coachTypeToIDMap {
		values = append(values, value)
	}
	sort.Strings(values)

	result := make([]models.CoachTypeFilterResponse, 0)
	for _, value := range values {
		coachTypeID := coachTypeToIDMap[value]
		availableVariants := f.variantStorage.GetAvailableVariantsByOption(coachTypeID)
		result = append(result, models.CoachTypeFilterResponse{
			Value:        value,
			Available:    len(availableVariants) > 0,
			Selected:     f.variantStorage.IsSelectedOption(coachTypeID),
			MinimumPrice: serialization.DumpPrice(getMinPrice(availableVariants)),
		})
	}
	return result
}

func (f *CoachTypeFilter) GetSearchParams() map[string][]string {
	const searchParamKey = "trainTariffClass"

	params := make(map[string][]string)
	for coachTypeName, coachTypeID := range coachTypeToIDMap {
		if f.variantStorage.IsSelectedOption(coachTypeID) {
			params[searchParamKey] = append(params[searchParamKey], coachTypeName)
		}
	}
	return params
}
