package filters

import (
	"encoding/json"
	"fmt"
	"os"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type Factory struct {
	logger                         log.Logger
	translatableFactory            *i18n.TranslatableFactory
	deluxeTrainNameFormsToID       map[string]int
	deluxeTrainNamePluralFormsToID map[int]string
}

type Config struct {
	TrainNameFormsPath       string `config:"direction-filters-trainnameforms-path,required"`
	TrainNamePluralFormsPath string `config:"direction-filters-trainnamepluralforms-path,required"`
}

var DefaultConfig = Config{
	TrainNameFormsPath:       "/app/train_name_forms_to_id.json",
	TrainNamePluralFormsPath: "/app/train_name_plural_forms_to_id.json",
}

type FactoryOption func(factory *Factory) *Factory

func WithConfigsLoading(logger log.Logger, cfg *Config) FactoryOption {
	return func(factory *Factory) *Factory {
		{
			file, err := os.Open(cfg.TrainNameFormsPath)
			if err != nil {
				logger.Fatal("json config not found",
					log.String("file_path", cfg.TrainNameFormsPath),
					log.Error(err),
				)
			}
			decoder := json.NewDecoder(file)
			if err := decoder.Decode(&factory.deluxeTrainNameFormsToID); err != nil {
				logger.Fatal("json config has unexpected format",
					log.String("file_path", cfg.TrainNameFormsPath),
					log.Error(err),
				)
			}
			logger.Info("config loaded", log.String("file_path", cfg.TrainNameFormsPath))
		}

		{
			file, err := os.Open(cfg.TrainNamePluralFormsPath)
			if err != nil {
				logger.Fatal("json config not found",
					log.String("file_path", cfg.TrainNamePluralFormsPath),
					log.Error(err),
				)
			}
			decoder := json.NewDecoder(file)
			if err := decoder.Decode(&factory.deluxeTrainNamePluralFormsToID); err != nil {
				logger.Fatal("json config has unexpected format",
					log.String("file_path", cfg.TrainNamePluralFormsPath),
					log.Error(err),
				)
			}
			logger.Info("config loaded", log.String("file_path", cfg.TrainNamePluralFormsPath))
		}
		return factory
	}
}

func NewFactory(logger log.Logger, translatableFactory *i18n.TranslatableFactory, options ...FactoryOption) *Factory {
	f := &Factory{
		logger:                         logger,
		translatableFactory:            translatableFactory,
		deluxeTrainNameFormsToID:       make(map[string]int),
		deluxeTrainNamePluralFormsToID: make(map[int]string),
	}
	for _, opt := range options {
		f = opt(f)
	}
	return f
}

func (f *Factory) Load(language lang.Lang, brandSelected []string, coachTypeSelected []string) (*Group, error) {
	const funcName = "direction.filters.Factory: Load"

	group := NewGroup()
	{
		brandFilter := NewBrandFilter(
			f.logger,
			language,
			f.translatableFactory,
			f.deluxeTrainNameFormsToID,
			f.deluxeTrainNamePluralFormsToID,
		)
		if err := brandFilter.LoadSelected(brandSelected); err != nil {
			return nil, fmt.Errorf("%s: %w", funcName, err)
		}
		group.AddFilter(brandFilter)
	}
	{
		coachTypeFilter := NewCoachTypeFilter()
		if err := coachTypeFilter.LoadSelected(coachTypeSelected); err != nil {
			return nil, fmt.Errorf("%s: %w", funcName, err)
		}
		group.AddFilter(coachTypeFilter)
	}

	for _, name := range []FilterName{
		placeCountName,
		facilityName,
		arrivalTimeName,
		priceName,
		departureTimeName,
	} {
		group.AddFakeFilterName(name)
	}
	return group, nil
}
