package filters

import (
	"context"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
)

type Filter interface {
	Name() FilterName
	Dump() interface{}
	BindVariants(variant ...segments.TrainVariant)
	IsSelectedVariant(segments.TrainVariant) bool
	MakeAvailableVariant(segments.TrainVariant)
	GetSearchParams() map[string][]string
}

type Group struct {
	filters         []Filter
	fakeFilterNames []FilterName
}

func NewGroup() *Group {
	return &Group{
		filters:         make([]Filter, 0),
		fakeFilterNames: make([]FilterName, 0),
	}
}

func (g *Group) AddFilter(filter Filter) {
	g.filters = append(g.filters, filter)
}

func (g *Group) AddFakeFilterName(name FilterName) {
	g.fakeFilterNames = append(g.fakeFilterNames, name)
}

func (g *Group) Apply(ctx context.Context, variants segments.TrainVariants) segments.TrainVariants {
	span, _ := opentracing.StartSpanFromContext(ctx, groupApplyCaller.String())
	defer span.Finish()

	var result segments.TrainVariants
	var selectors [][]bool

	for _, filter := range g.filters {
		filter.BindVariants(variants...)
		selectors = append(selectors, getSelected(filter, variants))
	}

	fullSelector := getFullSelector(len(variants))
	for filterNo, filter := range g.filters {
		filterAvailability := intersectSelectors(fullSelector, selectors[:filterNo]...)
		filterAvailability = intersectSelectors(filterAvailability, selectors[filterNo+1:]...)

		for i, v := range variants {
			if filterAvailability[i] {
				filter.MakeAvailableVariant(v)
			}
		}
	}

	for i, filtered := range intersectSelectors(fullSelector, selectors...) {
		if filtered {
			result = append(result, variants[i])
		}
	}
	return result
}

func getSelected(filter Filter, variants segments.TrainVariants) []bool {
	filterSelector := make([]bool, len(variants))
	for i, v := range variants {
		filterSelector[i] = filter.IsSelectedVariant(v)
	}

	for _, selected := range filterSelector {
		if selected {
			return filterSelector
		}
	}
	return getFullSelector(len(filterSelector))
}

func intersectSelectors(startAvailability []bool, availabilities ...[]bool) []bool {
	result := make([]bool, len(startAvailability))
	copy(result, startAvailability)

	for _, availability := range availabilities {
		for i, available := range availability {
			result[i] = result[i] && available
		}
	}
	return result
}

func (g *Group) Dump() map[string]interface{} {
	result := make(map[string]interface{})
	for _, filter := range g.filters {
		result[string(filter.Name())] = filter.Dump()
	}

	for _, fakeFilterName := range g.fakeFilterNames {
		result[string(fakeFilterName)] = []struct{}{}
	}
	return result
}

func (g *Group) GetSearchParams() map[string][]string {
	params := make(map[string][]string)
	for _, filter := range g.filters {
		for key, values := range filter.GetSearchParams() {
			params[key] = append(params[key], values...)
		}
	}
	return params
}

func (g *Group) GetBrandTitle() string {
	for _, filter := range g.filters {
		if filter.Name() == brandName {
			if brandFilter, ok := filter.(*BrandFilter); ok {
				return brandFilter.GetBrandTitle()
			}
		}
	}
	return ""
}
