package models

import "encoding/json"

type PriceResponse struct {
	Value    float64 `json:"value"`
	Currency string  `json:"currency"`
}

type URLResponse struct {
	Text     string `json:"text"`
	TouchURL string `json:"touch_url"`
	URL      string `json:"url"`
}

type BrandResponse struct {
	ID          int    `json:"id"`
	Title       string `json:"title"`
	ShortTitle  string `json:"short_title"`
	IsDeluxe    bool   `json:"is_deluxe"`
	IsHighSpeed bool   `json:"is_high_speed"`
}

type TrainResponse struct {
	Number            string         `json:"number"`
	DisplayNumber     string         `json:"display_number"`
	HasDynamicPricing bool           `json:"has_dynamic_pricing"`
	TwoStorey         bool           `json:"two_storey"`
	IsSuburban        bool           `json:"is_suburban"`
	CoachOwners       []string       `json:"coach_owners"`
	Title             string         `json:"title"`
	Brand             *BrandResponse `json:"brand"`
	ThreadType        string         `json:"thread_type"`
	FirstCountryCode  string         `json:"first_country_code"`
	LastCountryCode   string         `json:"last_country_code"`
	Provider          *string        `json:"provider"`
	RawTrainName      *string        `json:"raw_train_name"`
}

type PointResponse struct {
	Key   string `json:"key"`
	Title string `json:"title"`
}

type DateTimeResponse struct {
	Value    string `json:"value"`
	TimeZone string `json:"timezone"`
}

type DestinationResponse struct {
	Station       *PointResponse   `json:"station"`
	Settlement    *PointResponse   `json:"settlement"`
	LocalDatetime DateTimeResponse `json:"local_datetime"`
}

type PriceDetailResponse struct {
	Fee           string `json:"fee"`
	SeveralPrices bool   `json:"several_prices"`
	ServicePrice  string `json:"service_price"`
	TicketPrice   string `json:"ticket_price"`
}

type PlaceRecordsResponse struct {
	CoachType              string              `json:"coach_type"`
	ServiceClass           string              `json:"service_class"`
	Count                  int                 `json:"count"`
	LowerCount             int                 `json:"lower_count"`
	UpperCount             int                 `json:"upper_count"`
	LowerSideCount         int                 `json:"lower_side_count"`
	UpperSideCount         int                 `json:"upper_side_count"`
	MaxSeatsInTheSameCar   int                 `json:"max_seats_in_the_same_car"`
	HasNonRefundableTariff bool                `json:"has_non_refundable_tariff"`
	Price                  *PriceResponse      `json:"price"`
	PriceDetails           PriceDetailResponse `json:"price_details"`
}

type PlacesResponse struct {
	Records          []PlaceRecordsResponse `json:"records"`
	UpdatedAt        *DateTimeResponse      `json:"updated_at"`
	ElectronicTicket *bool                  `json:"electronic_ticket,omitempty"`
}

type BrokenClassesResponse map[string][]uint32

type SegmentResponse struct {
	Train         TrainResponse         `json:"train"`
	Departure     DestinationResponse   `json:"departure"`
	Arrival       DestinationResponse   `json:"arrival"`
	Duration      json.Number           `json:"duration"`
	Places        PlacesResponse        `json:"places"`
	BrokenClasses BrokenClassesResponse `json:"broken_classes"`
	Facilities    *bool                 `json:"facilities"`
	MinimumPrice  *PriceResponse        `json:"minimum_price"`
	OrderTouchURL string                `json:"order_touch_url"`
	OrderURL      string                `json:"order_url"`
	IsTheFastest  bool                  `json:"is_the_fastest"`
	IsTheCheapest bool                  `json:"is_the_cheapest"`
}

type QueryResponse struct {
	DeparturePoint *PointResponse `json:"departure_point"`
	ArrivalPoint   *PointResponse `json:"arrival_point"`
	DepartureDate  *string        `json:"departure_date"`
	Language       string         `json:"language"`
	OrderBy        string         `json:"order_by"`
}

type TitleResponse struct {
	HL string `json:"__hl"`
}

type DirectionResponse struct {
	Title              TitleResponse          `json:"title"`
	WizardReqID        *string                `json:"wizard_req_id,omitempty"`
	FoundDepartureDate string                 `json:"found_departure_date"`
	MinimumPrice       *PriceResponse         `json:"minimum_price"`
	MinimumDuration    *json.Number           `json:"minimum_duration"`
	PathItems          []URLResponse          `json:"path_items"`
	SearchTouchURL     string                 `json:"search_touch_url"`
	SearchURL          string                 `json:"search_url"`
	Total              int                    `json:"total"`
	SearchProps        map[string]interface{} `json:"search_props"`
	Segments           []*SegmentResponse     `json:"segments"`
	Query              QueryResponse          `json:"query"`
	Filters            map[string]interface{} `json:"filters"`
	ErrorCode          *int                   `json:"error_code,omitempty"`
	ErrorMsg           *string                `json:"error,omitempty"`
}

func (r *DirectionResponse) IsError() bool {
	if r.ErrorCode == nil {
		return false
	}
	return ResponseCode(*r.ErrorCode) == ErrorCode
}

func (r *DirectionResponse) IsEmpty() bool {
	if r.ErrorCode == nil {
		return false
	}
	return ResponseCode(*r.ErrorCode) == EmptyCode
}

func (r *DirectionResponse) ErrMsg() string {
	if r.ErrorMsg != nil {
		return *r.ErrorMsg
	}
	return ""
}
