package direction

import (
	"context"
	"encoding/json"
	"fmt"
	"html"
	"math"
	"reflect"
	"strconv"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/library/go/units"
	tpb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"

	api "a.yandex-team.ru/travel/trains/search_api/api/tariffs"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/eventdate"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/filters"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/query"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/serialization"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/sorting"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/consts"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/experiments"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/express"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/railway"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/searchprops"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/tariffs"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/threadtitle"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/url"
)

const (
	includeSearchProps    = true
	notIncludeSearchProps = false
)

type Provider struct {
	cfg                  *Config
	logger               log.Logger
	pointParser          *points.Parser
	repoRegistry         *registry.RepositoryRegistry
	scheduleRepository   *schedule.Repository
	tariffSelector       tariffs.TrainTariffSelector
	tariffMapper         *segments.Mapper
	urlFactory           *url.Factory
	filtersFactory       *filters.Factory
	expressRepository    *express.Repository
	trainTitleTranslator *i18n.TrainTitleTranslator
	timeTranslator       *i18n.TimeTranslator
	translatableFactory  *i18n.TranslatableFactory
	threadTitleGenerator *threadtitle.Generator
}

func NewProvider(
	cfg *Config,
	logger log.Logger,
	pointParser *points.Parser,
	repoRegistry *registry.RepositoryRegistry,
	scheduleRepository *schedule.Repository,
	tariffSelector tariffs.TrainTariffSelector,
	filtersFactory *filters.Factory,
	expressRepository *express.Repository,
	trainTitleTranslator *i18n.TrainTitleTranslator,
	timeTranslator *i18n.TimeTranslator,
	translatableFactory *i18n.TranslatableFactory,
) *Provider {
	return &Provider{
		cfg:                  cfg,
		logger:               logger,
		pointParser:          pointParser,
		repoRegistry:         repoRegistry,
		scheduleRepository:   scheduleRepository,
		tariffSelector:       tariffSelector,
		filtersFactory:       filtersFactory,
		expressRepository:    expressRepository,
		tariffMapper:         segments.NewMapper(logger, repoRegistry),
		urlFactory:           url.NewFactory(cfg.TravelURL),
		trainTitleTranslator: trainTitleTranslator,
		timeTranslator:       timeTranslator,
		translatableFactory:  translatableFactory,
		threadTitleGenerator: threadtitle.NewGenerator(
			logger,
			repoRegistry,
			translatableFactory,
			trainTitleTranslator,
		),
	}
}

func (p *Provider) GetDirections(ctx context.Context, rawQuery *query.RawDirectionQuery) (models.Response, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, providerGetDirectionAPIFn.String())
	defer span.Finish()

	if response, err := p.getDirections(ctx, rawQuery, includeSearchProps); err != nil {
		return nil, providerGetDirectionAPIFn.WrapError(err)
	} else {
		return response, nil
	}
}

func (p *Provider) GetOpenDirections(ctx context.Context, rawQuery *query.RawDirectionQuery) (models.Response, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, providerGetOpenDirectionAPIFn.String())
	defer span.Finish()

	if response, err := p.getDirections(ctx, rawQuery, notIncludeSearchProps); err != nil {
		return nil, providerGetOpenDirectionAPIFn.WrapError(err)
	} else {
		return response, nil
	}
}

func (p *Provider) getDirections(
	ctx context.Context,
	rawQuery *query.RawDirectionQuery,
	withSearchProps bool,
) (models.Response, error) {
	ctx = searchprops.With(ctx)
	ctx = experiments.ParseExperiments(ctx, rawQuery.ExpFlags)

	directionQuery, errResponse := p.buildDirectionQuery(ctx, rawQuery)
	if errResponse != nil {
		return errResponse, nil
	}

	var eventDateQuery eventdate.Query
	if directionQuery.DepartureDate != nil {
		eventDateQuery = eventdate.BuildFixedDateQuery(*directionQuery.DepartureDate, directionQuery.DepartureLocation)
	} else {
		eventDateQuery = eventdate.BuildFirstFutureDateQuery(directionQuery.DepartureLocation)
	}

	scheduleSegments := p.scheduleRepository.FindSegments(ctx, directionQuery.DeparturePoint, directionQuery.ArrivalPoint, eventDateQuery.GetMinDate())
	scheduleSegments = eventDateQuery.Filter(scheduleSegments)

	p.logger.Info(providerGetDirectionsFn.String(),
		log.String("departurePointKey", directionQuery.DeparturePoint.PointKey()),
		log.String("arrivalPointKey", directionQuery.ArrivalPoint.PointKey()),
		log.String("minEventDate", eventDateQuery.GetMinDate().String()),
		log.Int("foundSegmentsNumber", len(scheduleSegments)),
	)

	var foundDepartureDate time.Time
	var trainSegments segments.TrainSegments
	response := new(models.DirectionResponse)
	searchprops.Set(ctx, "train_common_wizard", "0")
	searchprops.Set(ctx, "train_pp_wizard", "1")
	searchprops.Set(ctx, "train_wizard_type", "undefined")
	searchprops.Set(ctx, "train_wizard_api_timeout", 0)

	if len(scheduleSegments) == 0 {
		trainSegments = make(segments.TrainSegments, 0)
		foundDepartureDate = eventDateQuery.GetMinDate()
		response.ErrorCode = ptr.Int(int(models.EmptyCode))
		response.ErrorMsg = ptr.String("Empty result")
	} else {
		foundDepartureDate = p.getDepartureDate(scheduleSegments[0])

		var err error
		trainSegments, err = p.makeTrainSegments(ctx, foundDepartureDate, directionQuery.DeparturePoint, directionQuery.ArrivalPoint, scheduleSegments)
		if err != nil {
			return nil, providerGetDirectionsFn.WrapError(err)
		}
		searchprops.Set(ctx, "train_wizard_type", "pp")
	}

	foundTrainSegmentNumber := len(trainSegments)
	allMinPrice := findMinPrice(trainSegments)
	allMinDuration := findMinDuration(trainSegments)

	{
		trainVariants := segments.Split(ctx, trainSegments)
		trainVariants = directionQuery.FilterGroup.Apply(ctx, trainVariants)
		trainVariants = directionQuery.Sorting.Sort(ctx, trainVariants)
		trainSegments = segments.Join(ctx, trainVariants)
	}

	if foundTrainSegmentNumber > 1 {
		setTheCheapest(allMinPrice, trainSegments)
		setTheFastest(allMinDuration, trainSegments)
	}

	spanDumpResponse, _ := opentracing.StartSpanFromContext(ctx, fmt.Sprintf("%s:dumpResponse", providerGetDirectionsFn))
	defer spanDumpResponse.Finish()

	response.FoundDepartureDate = foundDepartureDate.Format(date.DateISOFormat)
	serviceURL := p.urlFactory.MakeServiceURL(rawQuery.TLD, rawQuery.MainReqID)
	searchURL := p.urlFactory.MakeSearchURL(
		directionQuery.DeparturePoint,
		directionQuery.ArrivalPoint,
		response.FoundDepartureDate,
		rawQuery.TLD,
		len(scheduleSegments) > 0,
		rawQuery.MainReqID,
		directionQuery.FilterGroup.GetSearchParams(),
	)

	minDuration := findMinDuration(trainSegments)
	if minDuration != nil {
		response.MinimumDuration = dumpDuration(*minDuration)
	}
	response.MinimumPrice = serialization.DumpPrice(findMinPrice(trainSegments))
	response.SearchTouchURL = searchURL
	response.SearchURL = searchURL
	response.Total = len(trainSegments)

	var err error
	if response.PathItems, err = p.buildURLResponses(directionQuery, serviceURL, searchURL); err != nil {
		return nil, providerGetDirectionsFn.WrapError(err)
	}

	response.Query = p.buildQueryResponse(directionQuery)
	response.WizardReqID = rawQuery.MainReqID

	orderURLContext := p.makeOrderContextURL(directionQuery, response.FoundDepartureDate)
	response.Segments = make([]*models.SegmentResponse, len(trainSegments))
	for i, s := range trainSegments {
		response.Segments[i], err = p.buildSegmentResponse(directionQuery.Language, s, orderURLContext)
		if err != nil {
			return nil, providerGetDirectionsFn.WrapError(err)
		}
	}
	if withSearchProps {
		response.SearchProps = searchprops.GetMap(ctx)
	}
	response.Filters = directionQuery.FilterGroup.Dump()

	if titleResponse, err := p.buildTitleResponse(directionQuery, directionQuery.FilterGroup.GetBrandTitle()); err == nil {
		response.Title = titleResponse
	} else {
		return nil, providerGetDirectionsFn.WrapError(err)
	}

	return response, nil
}

func findMinPrice(segments segments.TrainSegments) *tpb.TPrice {
	if len(segments) == 0 {
		return nil
	}

	minPrice := segments[0].GetMinPrice()
	for i := 1; i < len(segments); i++ {
		segmentMinPrice := segments[i].GetMinPrice()
		if helpers.GetMinPrice(minPrice, segmentMinPrice) != minPrice {
			minPrice = segmentMinPrice
		}
	}
	return minPrice
}

func findMinDuration(segments segments.TrainSegments) *time.Duration {
	if len(segments) == 0 {
		return nil
	}

	minDuration := segments[0].Duration
	for i := 1; i < len(segments); i++ {
		segmentDuration := segments[i].Duration
		if segmentDuration < minDuration {
			minDuration = segmentDuration
		}
	}
	return &minDuration
}

func setTheCheapest(minPrice *tpb.TPrice, trainSegments segments.TrainSegments) {
	if minPrice != nil {
		for _, segment := range trainSegments {
			if helpers.PriceIsEqualOrNil(minPrice, segment.GetMinPrice()) {
				segment.IsTheCheapest = true
				break
			}
		}
	}
}

func setTheFastest(minDuration *time.Duration, trainSegments segments.TrainSegments) {
	if minDuration != nil {
		for _, segment := range trainSegments {
			if segment.Duration == *minDuration {
				segment.IsTheFastest = true
				break
			}
		}
	}
}

func (p *Provider) buildDirectionQuery(ctx context.Context, rawQuery *query.RawDirectionQuery) (*query.DirectionQuery, models.Response) {
	departurePoint, arrivalPoint, errResponse := query.ParsePoints(
		ctx,
		p.pointParser,
		rawQuery.DeparturePointKey,
		rawQuery.DepartureSettlementGeoID,
		rawQuery.ArrivalPointKey,
		rawQuery.ArrivalSettlementGeoID,
	)
	if errResponse != nil {
		return nil, errResponse
	}

	departureLocation, tzFound := p.repoRegistry.GetTimeZoneRepo().GetLocationByID(departurePoint.TimeZoneID())
	if !tzFound {
		departureLocation = consts.DefaultLocation
	}
	departureDate, errResponse := query.ParseDate(rawQuery.DepartureDate, departureLocation)
	if errResponse != nil {
		return nil, errResponse
	}

	language, errResponse := query.ParseLanguage(rawQuery.Language, p.cfg.DefaultLanguage, p.cfg.FrontendLanguages)
	if errResponse != nil {
		return nil, errResponse
	}

	filterGroup, err := p.filtersFactory.Load(language, rawQuery.BrandFilter, rawQuery.CoachTypeFilter)
	if err != nil {
		return nil, models.ResponseFromError(models.ErrorCode, err)
	}

	sorter, err := sorting.Load(rawQuery.OrderBy)
	if err != nil {
		return nil, models.ResponseFromError(models.ErrorCode, err)
	}

	return &query.DirectionQuery{
		DeparturePoint:    departurePoint,
		ArrivalPoint:      arrivalPoint,
		DepartureDate:     departureDate,
		DepartureLocation: departureLocation,
		MainReqID:         rawQuery.MainReqID,
		Language:          language,
		Sorting:           sorter,
		FilterGroup:       filterGroup,
		TLD:               rawQuery.TLD,
	}, nil
}

func (p *Provider) makeTrainSegments(ctx context.Context, foundDepartureDate time.Time, departurePoint, arrivalPoint points.Point, scheduleSegments schedule.Segments) (trainSegments segments.TrainSegments, err error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, providerMakeTrainSegmentsFn.String())
	defer span.Finish()

	departureExpressID := int32(p.expressRepository.FindExpressID(departurePoint))
	arrivalExpressID := int32(p.expressRepository.FindExpressID(arrivalPoint))

	var tariffInfos []*api.DirectionTariffInfo
	if departureExpressID == 0 || arrivalExpressID == 0 {
		p.logger.Warnf("Can not find prices by [%d-%d], because can not find express codes for one of points [%d-%d]",
			departureExpressID, arrivalExpressID,
			departurePoint.ID(), arrivalPoint.ID(),
		)
	} else {
		tariffInfos, err = p.tariffSelector.Select(
			ctx, []int32{departureExpressID}, []int32{arrivalExpressID},
			foundDepartureDate.Add(-units.Day), foundDepartureDate.Add(units.Day),
		)

		if err != nil {
			return nil, providerMakeTrainSegmentsFn.WrapError(err)
		}
	}

	railWayLocation := railway.GetLocationByPoint(departurePoint, p.repoRegistry)
	return p.tariffMapper.MapTrainSegments(ctx, railWayLocation, scheduleSegments, tariffInfos), nil
}

func (p *Provider) getDepartureDate(segment *schedule.Segment) time.Time {
	departureDate := segment.DepartureDateTime
	departureStationLocation := consts.DefaultLocation
	if departureStation, found := p.repoRegistry.GetStationRepo().Get(segment.Departure.StationId); found {
		if location, found := p.repoRegistry.GetTimeZoneRepo().GetLocationByID(departureStation.TimeZoneId); found {
			departureStationLocation = location
		}
	}
	return date.DateFromTime(departureDate.In(departureStationLocation))
}

func (p *Provider) buildQueryResponse(directionQuery *query.DirectionQuery) models.QueryResponse {
	departureDate := ""
	if directionQuery.DepartureDate != nil {
		departureDate = directionQuery.DepartureDate.Format(date.DateISOFormat)
	}

	dumped := models.QueryResponse{
		DeparturePoint: p.buildPointResponse(directionQuery.Language, directionQuery.DeparturePoint),
		ArrivalPoint:   p.buildPointResponse(directionQuery.Language, directionQuery.ArrivalPoint),
		DepartureDate:  helpers.OptString(departureDate),
		Language:       directionQuery.Language.String(),
		OrderBy:        directionQuery.Sorting.Name,
	}
	return dumped
}

func (p *Provider) buildPointResponse(language lang.Lang, point points.Point) *models.PointResponse {
	if point == nil || (reflect.ValueOf(point).Kind() == reflect.Ptr && reflect.ValueOf(point).IsNil()) {
		return nil
	}

	title, err := point.Translatable(p.translatableFactory).PopularTitle(language, lang.Nominative)
	if err != nil {
		p.logger.Warn("title not found", log.Error(err), log.String("point_key", point.PointKey()))
	}
	return &models.PointResponse{
		Key:   point.PointKey(),
		Title: title,
	}
}

func (p *Provider) buildURLResponses(directionQuery *query.DirectionQuery, serviceURL string, searchURL string) ([]models.URLResponse, error) {
	departureTitle, err := directionQuery.DeparturePoint.Translatable(p.translatableFactory).Title(directionQuery.Language, lang.Genitive)
	if err != nil {
		p.logger.Warn("title not found", log.Error(err), log.String("point_key", directionQuery.DeparturePoint.PointKey()))
	}

	arrivalTitle, err := directionQuery.ArrivalPoint.Translatable(p.translatableFactory).Title(directionQuery.Language, lang.Accusative)
	if err != nil {
		p.logger.Warn("title not found", log.Error(err), log.String("point_key", directionQuery.ArrivalPoint.PointKey()))
	}

	text, err := p.trainTitleTranslator.FormatTicketsWithPoints(directionQuery.Language, departureTitle, arrivalTitle)
	if err != nil {
		return nil, fmt.Errorf("%s: template processing fails: %w", providerBuildURLResponsesFn, err)
	}

	return []models.URLResponse{
		{
			Text:     p.urlFactory.MakeHost(directionQuery.TLD),
			TouchURL: serviceURL,
			URL:      serviceURL,
		},
		{
			Text:     text,
			TouchURL: searchURL,
			URL:      searchURL,
		},
	}, nil
}

func (p *Provider) buildSegmentResponse(language lang.Lang, trainSegment *segments.TrainSegment, orderURLContext url.OrderURLContext) (*models.SegmentResponse, error) {
	orderURL := orderURLContext.WithSegmentInfo(trainSegment.TrainNumber, trainSegment.Provider, trainSegment.DepartureLocalDt).String()
	trainTitle, err := p.threadTitleGenerator.FormatThreadTitle(language, trainSegment.TrainTitle)
	if err != nil {
		return nil, providerBuildSegmentResponseFn.WrapError(err)
	}
	return &models.SegmentResponse{
		Train: models.TrainResponse{
			Number:            trainSegment.TrainNumber,
			DisplayNumber:     trainSegment.DisplayNumber,
			HasDynamicPricing: trainSegment.HasDynamicPricing,
			TwoStorey:         trainSegment.TwoStorey,
			IsSuburban:        trainSegment.IsSuburban,
			CoachOwners:       trainSegment.CoachOwners,
			Title:             trainTitle,
			Brand:             p.buildBrandResponse(language, trainSegment.TrainBrand),
			ThreadType:        trainSegment.ThreadType,
			FirstCountryCode:  trainSegment.FirstCountryCode,
			LastCountryCode:   trainSegment.LastCountryCode,
			Provider:          helpers.OptString(trainSegment.Provider),
			RawTrainName:      helpers.OptString(trainSegment.RawTrainName),
		},
		Departure:     p.buildDestinationResponse(language, trainSegment.DepartureStation, trainSegment.DepartureLocalDt),
		Arrival:       p.buildDestinationResponse(language, trainSegment.ArrivalStation, trainSegment.ArrivalLocalDt),
		Places:        p.buildPlacesResponse(trainSegment),
		Duration:      *dumpDuration(trainSegment.Duration),
		BrokenClasses: buildBrokenClassesResponse(trainSegment.BrokenClasses),
		MinimumPrice:  serialization.DumpPrice(trainSegment.GetMinPrice()),
		OrderTouchURL: orderURL,
		OrderURL:      orderURL,
		IsTheCheapest: trainSegment.IsTheCheapest,
		IsTheFastest:  trainSegment.IsTheFastest,
	}, nil
}

func (p *Provider) buildDestinationResponse(language lang.Lang, threadStation *rasp.TThreadStation, dt time.Time) models.DestinationResponse {
	station, _ := p.repoRegistry.GetStationRepo().Get(threadStation.StationId)

	var settlement *rasp.TSettlement
	if station != nil {
		settlement, _ = p.repoRegistry.GetSettlementRepo().Get(station.SettlementId)
	}

	return models.DestinationResponse{
		Station:       p.buildPointResponse(language, points.NewStation(station)),
		Settlement:    p.buildPointResponse(language, points.NewSettlement(settlement)),
		LocalDatetime: *p.dumpDateTime(&dt),
	}
}

func buildBrokenClassesResponse(classes *api.TariffBrokenClasses) models.BrokenClassesResponse {
	if classes == nil {
		return nil
	}

	response := make(models.BrokenClassesResponse)
	for key, value := range map[string][]uint32{
		"unknown":     classes.Unknown,
		"soft":        classes.Soft,
		"platzkarte":  classes.Platzkarte,
		"compartment": classes.Compartment,
		"suite":       classes.Suite,
		"common":      classes.Common,
		"sitting":     classes.Sitting,
	} {
		if len(value) > 0 {
			response[key] = value
		}
	}
	return response
}

func (p *Provider) buildPlacesResponse(trainSegment *segments.TrainSegment) models.PlacesResponse {
	var electronicTicket *bool

	var records []models.PlaceRecordsResponse
	for _, place := range trainSegment.Places {
		records = append(records, models.PlaceRecordsResponse{
			CoachType:              place.CoachType,
			ServiceClass:           place.ServiceClass,
			Count:                  int(place.Count),
			LowerCount:             int(place.LowerCount),
			UpperCount:             int(place.UpperCount),
			LowerSideCount:         int(place.LowerSideCount),
			UpperSideCount:         int(place.UpperSideCount),
			MaxSeatsInTheSameCar:   int(place.MaxSeatsInTheSameCar),
			HasNonRefundableTariff: place.HasNonRefundableTariff,
			Price:                  serialization.DumpPrice(place.Price),
			PriceDetails:           p.buildPriceDetailResponse(place.PriceDetails),
		})
		if electronicTicket == nil && place.Count > 0 {
			electronicTicket = ptr.Bool(trainSegment.ElectronicTicket)
		}
	}

	return models.PlacesResponse{
		Records:          records,
		UpdatedAt:        p.dumpDateTime(trainSegment.UpdatedAt),
		ElectronicTicket: electronicTicket,
	}
}

func (p *Provider) dumpDateTime(dt *time.Time) *models.DateTimeResponse {
	if dt == nil {
		return nil
	}

	format := date.DateTimeISOFormat
	if dt.Nanosecond() != 0 {
		format = date.DateTimeISOWithNanoFormat
	}
	return &models.DateTimeResponse{
		Value:    dt.Format(format),
		TimeZone: dt.Location().String(),
	}
}

func (p *Provider) buildPriceDetailResponse(details *api.TrainPlacePriceDetails) models.PriceDetailResponse {
	return models.PriceDetailResponse{
		Fee:           getStringValueFromPrice(details.Fee),
		SeveralPrices: details.SeveralPrices,
		ServicePrice:  getStringValueFromPrice(details.ServicePrice),
		TicketPrice:   getStringValueFromPrice(details.TicketPrice),
	}
}

func (p *Provider) buildBrandResponse(language lang.Lang, namedTrain *rasp.TNamedTrain) *models.BrandResponse {
	if namedTrain == nil {
		return nil
	}
	translatableNamedTrain := p.translatableFactory.TranslatableNamedTrain(namedTrain)

	title, err := translatableNamedTrain.Title(language, lang.Nominative)
	if err != nil {
		p.logger.Warn("not found title for named train",
			log.Int32("namedTrainID", namedTrain.Id),
			log.Error(err),
		)
	}

	shortTitle, err := translatableNamedTrain.ShortTitle(language)
	if err != nil {
		p.logger.Warn("not found short title for named train",
			log.Int32("namedTrainID", namedTrain.Id),
			log.Error(err),
		)
	}

	return &models.BrandResponse{
		ID:          int(namedTrain.Id),
		Title:       title,
		ShortTitle:  shortTitle,
		IsDeluxe:    namedTrain.IsDeluxe,
		IsHighSpeed: namedTrain.IsHighSpeed,
	}
}

func (p *Provider) buildTitleResponse(directionQuery *query.DirectionQuery, brandTitle string) (models.TitleResponse, error) {
	departureTitle, err := directionQuery.DeparturePoint.Translatable(p.translatableFactory).Title(directionQuery.Language, lang.Nominative)
	if err != nil {
		p.logger.Warn("title not found", log.Error(err), log.String("point_key", directionQuery.DeparturePoint.PointKey()))
	}

	arrivalTitle, err := directionQuery.ArrivalPoint.Translatable(p.translatableFactory).Title(directionQuery.Language, lang.Nominative)
	if err != nil {
		p.logger.Warn("title not found", log.Error(err), log.String("point_key", directionQuery.ArrivalPoint.PointKey()))
	}

	var text string
	if len(brandTitle) != 0 {
		text, err = p.trainTitleTranslator.FormatBrandSchedule(directionQuery.Language, departureTitle, arrivalTitle, brandTitle)
	} else {
		text, err = p.trainTitleTranslator.FormatTrainsSchedule(directionQuery.Language, departureTitle, arrivalTitle)
	}

	if err != nil {
		return models.TitleResponse{}, providerBuildTitleResponseFn.WrapError(err)
	}

	if directionQuery.DepartureDate != nil {
		translatedDepartureDate, err := p.timeTranslator.FormatHumanDate(directionQuery.Language, *directionQuery.DepartureDate)
		if err != nil {
			return models.TitleResponse{}, providerBuildTitleResponseFn.WrapError(err)
		}
		text = fmt.Sprintf("%s, %s", text, translatedDepartureDate)
	}
	return models.TitleResponse{HL: html.UnescapeString(text)}, nil
}

func (p *Provider) makeOrderContextURL(directionQuery *query.DirectionQuery, departureDate string) url.OrderURLContext {
	extra := map[string][]string{
		"fromId":        {directionQuery.DeparturePoint.PointKey()},
		"toId":          {directionQuery.ArrivalPoint.PointKey()},
		"when":          {departureDate},
		"transportType": {"train"},
	}

	translatableDeparture := directionQuery.DeparturePoint.Translatable(p.translatableFactory)
	if title, err := translatableDeparture.Title(directionQuery.Language, lang.Nominative); err != nil {
		p.logger.Warn("translation not found",
			log.String("point_key", directionQuery.DeparturePoint.PointKey()),
			log.Error(err),
		)
	} else {
		extra["fromName"] = append(extra["fromName"], title)
	}

	translatableArrival := directionQuery.ArrivalPoint.Translatable(p.translatableFactory)
	if title, err := translatableArrival.Title(directionQuery.Language, lang.Nominative); err != nil {
		p.logger.Warn("translation not found",
			log.String("point_key", directionQuery.ArrivalPoint.PointKey()),
			log.Error(err),
		)
	} else {
		extra["toName"] = append(extra["fromName"], title)
	}

	if directionQuery.MainReqID != nil {
		extra["wizardReqId"] = append(extra["wizardReqId"], *directionQuery.MainReqID)
	}

	return p.urlFactory.MakeOrderContextURL(directionQuery.TLD, extra)
}

func getStringValueFromPrice(price *tpb.TPrice) string {
	value := float64(price.Amount) / math.Pow10(int(price.Precision))
	prec := -1
	if value == math.Round(value) {
		prec = 1
	}
	return strconv.FormatFloat(value, 'f', prec, 64)
}

func dumpDuration(d time.Duration) *json.Number {
	result := json.Number(fmt.Sprintf("%.1f", float32(d.Seconds())/60))
	return &result
}
