package query

import (
	"context"
	"fmt"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/clock"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
)

func ParseDate(rawDate string, location *time.Location) (*time.Time, *models.ErrorResponse) {
	if len(rawDate) == 0 {
		return nil, nil
	}

	parsed, err := time.Parse(date.DateISOFormat, rawDate)
	if err != nil {
		return nil, models.ResponseFromError(
			models.ErrorCode,
			fmt.Errorf("%s: %w, raw_date=%s", parseDateFnCaller, InvalidDateFormatError, rawDate),
		)
	}

	today := date.DateFromTime(clock.Now().In(location))
	if parsed.Before(today) {
		return nil, models.ResponseFromError(
			models.ErrorCode,
			fmt.Errorf("%s: %w, raw_date=%s", parseDateFnCaller, PastDateError, rawDate),
		)
	}

	if today.Add(time.Duration(futureDaysLimit) * units.Day).Before(parsed) {
		return nil, models.ResponseFromError(
			models.ErrorCode,
			fmt.Errorf("%s: %w, raw_date=%s", parseDateFnCaller, FutureDateError, rawDate),
		)
	}

	return &parsed, nil
}

func ParseLanguage(rawLanguage string, defaultLang lang.Lang, frontendLanguages map[lang.Lang]bool) (lang.Lang, *models.ErrorResponse) {
	if len(rawLanguage) == 0 {
		return defaultLang, nil
	}

	language := lang.Lang(rawLanguage)
	if frontendLanguages[language] {
		return language, nil
	}
	var languages []string
	for l := range frontendLanguages {
		languages = append(languages, string(l))
	}
	return "", models.ResponseFromError(
		models.ErrorCode,
		fmt.Errorf("%s: invalid language value, it should be one of [%s]", parseLanguageFnCaller, languages),
	)
}

func ParsePoints(
	ctx context.Context,
	p *points.Parser,
	departurePointKey string,
	departureSettlementGeoID string,
	arrivalPointKey string,
	arrivalSettlementGeoID string,
) (points.Point, points.Point, *models.ErrorResponse) {
	span, _ := opentracing.StartSpanFromContext(ctx, parsePointsFnCaller.String())
	defer span.Finish()

	departurePoint, err := p.ParsePoint(departurePointKey, departureSettlementGeoID)
	if err != nil {
		return nil, nil, models.ResponseFromError(models.ErrorCode, withPointType(err, departure))
	}

	arrivalPoint, err := p.ParsePoint(arrivalPointKey, arrivalSettlementGeoID)
	if err != nil {
		return nil, nil, models.ResponseFromError(models.ErrorCode, withPointType(err, arrival))
	}

	if points.Equal(departurePoint, arrivalPoint) {
		return nil, nil, models.ResponseFromError(
			models.ErrorCode,
			fmt.Errorf("%s: %w", parsePointsFnCaller, EqualPointsError),
		)
	}

	for _, point := range []points.Point{departurePoint, arrivalPoint} {
		if _, found := CountriesWhereSaleTicketsIds[point.CountryID()]; !found {
			return nil, nil, models.ResponseFromError(
				models.EmptyCode,
				fmt.Errorf("%s: %w", parsePointsFnCaller, ProhibitedSalesError),
			)
		}
	}

	if (departurePoint.RegionID() == kaliningradRegionID) != (arrivalPoint.RegionID() == kaliningradRegionID) {
		return nil, nil, models.ResponseFromError(
			models.EmptyCode,
			fmt.Errorf("%s: %w", parsePointsFnCaller, CrossingKaliningradBorderError),
		)
	}
	return departurePoint, arrivalPoint, nil
}

func withPointType(err error, pointType pointType) error {
	return fmt.Errorf("%s: point_type=%s: %w", parsePointsFnCaller, pointType, err)
}
