package segments

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/library/go/funcnames"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/api/tariffs"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/mappers"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
)

var (
	mapperMapTrainSegmentsCaller funcnames.Caller
	splitCaller                  funcnames.Caller
	joinCaller                   funcnames.Caller
)

func init() {
	mapperMapTrainSegmentsCaller = funcnames.BuildFullName((&Mapper{}).MapTrainSegments)
	splitCaller = funcnames.BuildFullName(Split)
	joinCaller = funcnames.BuildFullName(Join)
}

type Mapper struct {
	logger       log.Logger
	repoRegistry *registry.RepositoryRegistry
}

func NewMapper(logger log.Logger, repoRegistry *registry.RepositoryRegistry) *Mapper {
	return &Mapper{
		logger:       logger,
		repoRegistry: repoRegistry,
	}
}

func (m *Mapper) MapTrainSegments(
	ctx context.Context,
	railWayLocation *time.Location,
	scheduleSegments schedule.Segments,
	tariffSegmentsInfos []*tariffs.DirectionTariffInfo,
) TrainSegments {
	span, _ := opentracing.StartSpanFromContext(ctx, mapperMapTrainSegmentsCaller.String())
	defer span.Finish()

	tariffSegments := makeTariffSegmentsMap(tariffSegmentsInfos)
	updatedInfo := makeUpdatedInfo(railWayLocation, tariffSegmentsInfos)

	var result TrainSegments
	for _, segment := range scheduleSegments {
		trainNumber := segment.Thread.Number
		localDepartureDateTime := m.localizeByStation(segment.DepartureDateTime, segment.Departure)
		tariffSegment := tariffSegments.get(TariffSegmentKey{
			DepartureStationID: segment.Departure.StationId,
			ArrivalStationID:   segment.Arrival.StationId,
			Number:             trainNumber,
			DepartureDateTime:  segment.DepartureDateTime.UTC(),
		})
		train := &TrainSegment{
			TrainNumber:      trainNumber,
			TrainTitle:       segment.Thread.CommonTitle,
			ThreadType:       mappers.GetThreadTypeCode(segment.Thread.Type),
			Duration:         segment.ArrivalDateTime.Sub(segment.DepartureDateTime),
			DepartureStation: segment.Departure,
			DepartureLocalDt: localDepartureDateTime,
			ArrivalStation:   segment.Arrival,
			ArrivalLocalDt:   m.localizeByStation(segment.ArrivalDateTime, segment.Arrival),
		}
		if brand, found := m.repoRegistry.GetNamedTrainRepo().GetByNumber(trainNumber); found {
			train.TrainBrand = brand
		}

		if tariffSegment != nil {
			train.ElectronicTicket = tariffSegment.ElectronicTicket
			train.BrokenClasses = tariffSegment.BrokenClasses
			train.DisplayNumber = tariffSegment.DisplayNumber
			train.HasDynamicPricing = tariffSegment.HasDynamicPricing
			train.TwoStorey = tariffSegment.TwoStorey
			train.IsSuburban = tariffSegment.IsSuburban
			train.CoachOwners = tariffSegment.CoachOwners
			train.FirstCountryCode = tariffSegment.FirstCountryCode
			train.LastCountryCode = tariffSegment.LastCountryCode
			train.Provider = tariffSegment.Provider
			train.RawTrainName = tariffSegment.RawTrainName
			train.Places = tariffSegment.Places
			train.UpdatedAt = getUpdatedAt(railWayLocation, segment.DepartureDateTime, updatedInfo)
		}

		result = append(result, train)
	}
	return result
}

func (m *Mapper) localizeByStation(d time.Time, threadStation *rasp.TThreadStation) time.Time {
	if station, found := m.repoRegistry.GetStationRepo().Get(threadStation.StationId); found {
		return m.repoRegistry.GetTimeZoneRepo().Localize(d, station.TimeZoneId)
	}
	return d
}

type TariffSegmentKey struct {
	DepartureStationID int32
	ArrivalStationID   int32
	Number             string
	DepartureDateTime  time.Time
}
type SegmentToTariffsMap map[TariffSegmentKey]*tariffs.DirectionTariffTrain

func makeTariffSegmentsMap(infos []*tariffs.DirectionTariffInfo) SegmentToTariffsMap {
	result := make(SegmentToTariffsMap)
	for _, tariffInfo := range infos {
		for _, segment := range tariffInfo.Data {
			key := TariffSegmentKey{
				DepartureStationID: segment.DepartureStationId,
				ArrivalStationID:   segment.ArrivalStationId,
				Number:             segment.Number,
				DepartureDateTime:  segment.Departure.AsTime(),
			}
			result[key] = segment
		}
	}
	return result
}

func (m SegmentToTariffsMap) get(key TariffSegmentKey) *tariffs.DirectionTariffTrain {
	if segment, found := m[key]; found {
		return segment
	}
	return nil
}

func makeUpdatedInfo(railWayLocation *time.Location, infos []*tariffs.DirectionTariffInfo) map[time.Time]time.Time {
	if railWayLocation == nil {
		railWayLocation = time.UTC
	}

	updatedInfo := make(map[time.Time]time.Time)
	for _, info := range infos {
		if info.DepartureDate == nil {
			continue
		}

		// ключ - начало суток полученное от DepartureDate в railWay локации
		key := setLocation(date.GetDateFromProto(info.DepartureDate).Truncate(units.Day), railWayLocation)
		updatedInfo[key] = info.UpdatedAt.AsTime().UTC()
	}
	return updatedInfo
}

func getUpdatedAt(railWayLocation *time.Location, departureDate time.Time, updatedInfo map[time.Time]time.Time) *time.Time {
	if railWayLocation == nil {
		railWayLocation = time.UTC
	}

	// обрезаем дату до начало суток в railWay локации
	y, m, d := departureDate.Date()
	key := time.Date(y, m, d, 0, 0, 0, 0, railWayLocation)
	if key.After(departureDate) {
		key = key.Add(-units.Day)
	}

	if updatedAt, found := updatedInfo[key]; found {
		return &updatedAt
	}
	return nil
}

func setLocation(dt time.Time, l *time.Location) time.Time {
	y, m, d := dt.Date()
	return time.Date(y, m, d, dt.Hour(), dt.Minute(), dt.Second(), dt.Nanosecond(), l)
}

func Split(ctx context.Context, segments TrainSegments) (variants TrainVariants) {
	span, _ := opentracing.StartSpanFromContext(ctx, splitCaller.String())
	defer span.Finish()

	for _, s := range segments {
		if len(s.Places) == 0 {
			variants = append(variants, TrainVariant{
				Segment: s,
				Place:   nil,
			})
		} else {
			for _, place := range s.Places {
				variants = append(variants, TrainVariant{
					Segment: s,
					Place:   place,
				})
			}
		}
	}
	return
}

func Join(ctx context.Context, variants TrainVariants) (segments TrainSegments) {
	span, _ := opentracing.StartSpanFromContext(ctx, joinCaller.String())
	defer span.Finish()

	placesByID := make(map[*TrainSegment]struct{})
	for _, v := range variants {
		if _, found := placesByID[v.Segment]; !found {
			placesByID[v.Segment] = struct{}{}
			segments = append(segments, v.Segment)
			v.Segment.Places = nil
		}
		if v.Place != nil {
			v.Segment.Places = append(v.Segment.Places, v.Place)
		}
	}
	return
}
