package sorting

import (
	"context"
	"fmt"
	"strings"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/library/go/funcnames"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
)

const defaultSorting = "best"

var sortersMap = map[string]SorterFn{
	"best":       ByBestSorter,
	"departure":  ByDepartureSorter,
	"arrival":    ByArrivalSorter,
	"duration":   ByDurationSorter,
	"price":      ByPriceSorter,
	"-departure": ByDepartureReverseSorter,
	"-arrival":   ByArrivalReverseSorter,
	"-duration":  ByDurationReverseSorter,
	"-price":     ByPriceReverseSorter,
}
var InvalidOrderError error
var sorterSortCaller funcnames.Caller

func init() {
	var names []string
	for name := range sortersMap {
		names = append(names, name)
	}
	InvalidOrderError = fmt.Errorf("invalid order_by value: it should be one of %s", strings.Join(names, ", "))

	sorterSortCaller = funcnames.BuildFullName((&Sorter{}).Sort)
}

type SorterFn func(variants segments.TrainVariants) segments.TrainVariants

type Sorter struct {
	Name     string
	sorterFn SorterFn
}

func (s *Sorter) Sort(ctx context.Context, variants segments.TrainVariants) segments.TrainVariants {
	span, _ := opentracing.StartSpanFromContext(ctx, sorterSortCaller.String())
	defer span.Finish()

	return s.sorterFn(variants)
}

func Load(value string) (Sorter, error) {
	if len(value) == 0 {
		value = defaultSorting
	}

	if s, found := sortersMap[value]; !found {
		return Sorter{}, InvalidOrderError
	} else {
		return Sorter{
			Name:     value,
			sorterFn: s,
		}, nil
	}
}
