package sorting

import (
	"math"
	"sort"

	tpb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
)

type SwappableVariants segments.TrainVariants

func (s SwappableVariants) Len() int      { return len(s) }
func (s SwappableVariants) Swap(i, j int) { s[i], s[j] = s[j], s[i] }

type ByBest struct {
	SwappableVariants
}

func ByBestSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Stable(ByBest{SwappableVariants(variants)})
	return variants
}

func (s ByBest) Less(i, j int) bool {
	var infPrice = &tpb.TPrice{
		Currency:  tpb.ECurrency_C_RUB,
		Amount:    math.MaxInt32,
		Precision: 2,
	}

	left, right := s.SwappableVariants[i], s.SwappableVariants[j]

	var leftPrice = infPrice
	if left.Place != nil && left.Place.Price != nil {
		leftPrice = left.Place.Price
	}

	var rightPrice = infPrice
	if right.Place != nil && right.Place.Price != nil {
		rightPrice = right.Place.Price
	}

	// дорогие поезда хуже дешёвых и поезда без цен хуже поездов с ценами
	if !helpers.PriceIsEqualOrNil(leftPrice, rightPrice) {
		return helpers.GetMinPrice(leftPrice, rightPrice) == leftPrice
	}

	// поезда, которые отправляются или прибывают ночью (00:00 - 06:00), хуже остальных
	isLeftNight := isNightVariant(left)
	if isLeftNight != isNightVariant(right) {
		return !isLeftNight
	}

	return left.Segment.Duration < right.Segment.Duration
}

func isNightVariant(v segments.TrainVariant) bool {
	return v.Segment.DepartureLocalDt.Hour() < 6 || v.Segment.ArrivalLocalDt.Hour() < 6
}

type ByDeparture struct {
	SwappableVariants
}

func ByDepartureSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(ByDeparture{SwappableVariants(variants)})
	return variants
}

func ByDepartureReverseSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(sort.Reverse(ByDeparture{SwappableVariants(variants)}))
	return variants
}

func (s ByDeparture) Less(i, j int) bool {
	left, right := s.SwappableVariants[i], s.SwappableVariants[j]
	return left.Segment.DepartureLocalDt.Before(right.Segment.DepartureLocalDt)
}

type ByArrival struct {
	SwappableVariants
}

func ByArrivalSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(ByArrival{SwappableVariants(variants)})
	return variants
}

func ByArrivalReverseSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(sort.Reverse(ByArrival{SwappableVariants(variants)}))
	return variants
}

func (s ByArrival) Less(i, j int) bool {
	left, right := s.SwappableVariants[i], s.SwappableVariants[j]
	return left.Segment.ArrivalLocalDt.Before(right.Segment.ArrivalLocalDt)
}

type ByDuration struct {
	SwappableVariants
}

func ByDurationSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(ByDuration{SwappableVariants(variants)})
	return variants
}

func ByDurationReverseSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(sort.Reverse(ByDuration{SwappableVariants(variants)}))
	return variants
}

func (s ByDuration) Less(i, j int) bool {
	left, right := s.SwappableVariants[i], s.SwappableVariants[j]
	return left.Segment.Duration < right.Segment.Duration
}

type ByPrice struct {
	SwappableVariants
}

func ByPriceSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(ByPrice{SwappableVariants(variants)})
	return variants
}

func ByPriceReverseSorter(variants segments.TrainVariants) segments.TrainVariants {
	sort.Sort(sort.Reverse(ByPrice{SwappableVariants(variants)}))
	return variants
}

func (s ByPrice) Less(i, j int) bool {
	left, right := s.SwappableVariants[i], s.SwappableVariants[j]

	var leftPrice *tpb.TPrice
	if left.Place != nil {
		leftPrice = left.Place.Price
	}

	var rightPrice *tpb.TPrice
	if right.Place != nil {
		rightPrice = right.Place.Price
	}

	return helpers.GetMinPrice(leftPrice, rightPrice) == leftPrice
}
