package handler

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	travelpb "a.yandex-team.ru/travel/proto"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes"

	commonModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/common/models"
	mbModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/mordabackend/models"

	apipb "a.yandex-team.ru/travel/trains/search_api/api"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
	"a.yandex-team.ru/travel/trains/search_api/internal/searcher/models"
)

type Converter struct {
	logger log.Logger
}

func NewConverter(logger log.Logger) *Converter {
	return &Converter{
		logger: logger,
	}
}

func (c *Converter) SearchResultToProto(searchResult *models.TransferVariantsResponse) *apipb.Response {
	const funcName = "Converter.searchResultToProto"

	resp := apipb.Response{
		Status:         apipb.ResponseStatus_STATUS_DONE,
		SearchContext:  searchContextToProto(searchResult.Context),
		ActivePartners: searchResult.ActivePartners,
		NearestDates:   nearestDatesToProto(searchResult.NearestDays),
	}
	if searchResult.Querying {
		resp.Status = apipb.ResponseStatus_STATUS_QUERYING
	}

	for _, variant := range searchResult.TransferVariants {
		forwardSegments, err := c.segmentsToProto(variant.Forward)
		if err != nil {
			c.logger.Errorf("%s: variant skipped: %s", funcName, err.Error())
			continue
		}
		backwardSegments, err := c.segmentsToProto(variant.Backward)
		if err != nil {
			c.logger.Errorf("%s: variant skipped: %s", funcName, err.Error())
			continue
		}
		resp.Variants = append(resp.Variants, &apipb.Variant{
			Id:       variant.ID,
			Forward:  forwardSegments,
			Backward: backwardSegments,
			MinPrice: c.priceToProto(variant.MinPrice),
			OrderUrl: orderURLToProto(variant.OrderURL),
		})
	}

	return &resp
}

func searchContextToProto(context mbModels.ResponseContext) *apipb.SearchContext {
	result := &apipb.SearchContext{
		IsChanged:      context.IsChanged,
		Original:       searchContextPointsToProto(context.Original),
		Search:         searchContextPointsToProto(context.Search),
		TransportTypes: context.TransportTypes,
	}
	result.LatestDatetime, _ = ptypes.TimestampProto(context.LatestDatetime)
	return result
}

func searchContextPointsToProto(points mbModels.ResponseSearchPoints) *apipb.SearchContextPoints {
	return &apipb.SearchContextPoints{
		Nearest:   points.Nearest,
		PointFrom: searchContextPointToProto(points.PointFrom),
		PointTo:   searchContextPointToProto(points.PointTo),
	}
}

func searchContextPointToProto(point mbModels.ResponseSearchPoint) *apipb.SearchContextPoint {
	result := &apipb.SearchContextPoint{
		Key:   point.PointKey,
		Title: point.Title,
		Slug:  point.Slug,
	}
	if point.TitleWithType != "" {
		result.OptionalTitleWithType = &apipb.SearchContextPoint_TitleWithType{TitleWithType: point.TitleWithType}
	}
	if point.TitleGenitive != "" {
		result.OptionalTitleGenitive = &apipb.SearchContextPoint_TitleGenitive{TitleGenitive: point.TitleGenitive}
	}
	if point.TitleAccusative != "" {
		result.OptionalTitleAccusative = &apipb.SearchContextPoint_TitleAccusative{TitleAccusative: point.TitleAccusative}
	}
	if point.TitleLocative != "" {
		result.OptionalTitleLocative = &apipb.SearchContextPoint_TitleLocative{TitleLocative: point.TitleLocative}
	}
	if point.Preposition != "" {
		result.OptionalPreposition = &apipb.SearchContextPoint_Preposition{Preposition: point.Preposition}
	}
	if point.PopularTitle != "" {
		result.OptionalPopularTitle = &apipb.SearchContextPoint_PopularTitle{PopularTitle: point.PopularTitle}
	}
	if point.ShortTitle != "" {
		result.OptionalShortTitle = &apipb.SearchContextPoint_ShortTitle{ShortTitle: point.ShortTitle}
	}
	return result
}

func (c *Converter) segmentsToProto(segments []commonModels.SegmentWithTariffs) ([]*apipb.Segment, error) {
	const funcName = "Converter.segmentsToProto"

	protoSegments := make([]*apipb.Segment, 0, len(segments))
	for _, segment := range segments {
		departure, err := ptypes.TimestampProto(segment.Departure)
		if err != nil {
			return nil, fmt.Errorf("%s: marshaling error: %w", funcName, err)
		}
		arrival, err := ptypes.TimestampProto(segment.Arrival)
		if err != nil {
			return nil, fmt.Errorf("%s: marshaling error: %w", funcName, err)
		}

		var companyTitle string
		for _, t := range []string{segment.Company.ShortTitle, segment.Company.Title, segment.Company.UFSTitle} {
			if t != "" {
				companyTitle = t
				break
			}
		}

		protoSegments = append(protoSegments, &apipb.Segment{
			Id:          segment.ID,
			Departure:   departure,
			Arrival:     arrival,
			StationFrom: stationToProto(&segment.StationFrom),
			StationTo:   stationToProto(&segment.StationTo),
			Company: &apipb.Company{
				Id:    uint32(segment.Company.ID),
				Title: companyTitle,
			},
			Train: &apipb.Train{
				Title:         segment.Title,
				Number:        segment.OriginalNumber,
				DisplayNumber: segment.Number,
			},
			Features: featuresToProto(&segment),
			Tariffs:  c.tariffsToProto(&segment.Tariffs),
			Provider: segment.Provider,
		})
	}
	return protoSegments, nil
}

func stationToProto(station *commonModels.Station) *apipb.Station {
	title := station.PopularTitle
	if title == "" {
		title = station.Title
	}
	return &apipb.Station{
		Id:    fmt.Sprintf("%d", station.ID),
		Title: title,
		Country: &apipb.Country{
			Id:   uint32(station.Country.ID),
			Code: station.Country.Code,
		},
		Settlement: &apipb.Settlement{
			Id:              uint32(station.Settlement.ID),
			Preposition:     station.Settlement.Preposition,
			Title:           station.Settlement.Title,
			TitleAccusative: station.Settlement.TitleAccusative,
			TitleGenitive:   station.Settlement.TitleGenitive,
			TitleLocative:   station.Settlement.TitleLocative,
		},
		Timezone:        station.Timezone,
		Platform:        station.Platform,
		RailwayTimezone: station.RailwayTimezone,
	}
}

func (c *Converter) classPriceToProto(classPrice *commonModels.ClassPrice) *apipb.MinTariffsClass {
	if *classPrice == (commonModels.ClassPrice{}) {
		return nil
	}

	return &apipb.MinTariffsClass{
		Price:                  c.priceToProto(classPrice.Price),
		Seats:                  uint32(classPrice.Seats),
		HasNonRefundableTariff: classPrice.HasNonRefundableTariff,
		PlacesDetails:          c.createPlacesDetails(classPrice),
	}
}

func (c *Converter) priceToProto(price commonModels.Price) *travelpb.TPrice {
	const (
		funcName = "Converter.priceToProto"
	)
	if price == (commonModels.Price{}) {
		return nil
	}
	protoPrice, err := helpers.PriceToProto(&price)
	if err != nil {
		c.logger.Errorf("%s: %v", funcName, err.Error())
		return nil
	}
	return protoPrice
}

func featuresToProto(segment *commonModels.SegmentWithTariffs) *apipb.Features {
	features := apipb.Features{
		Eticket:        segment.Tariffs.ElectronicTicket,
		DynamicPricing: segment.HasDynamicPricing,
		ThroughTrain:   segment.IsThroughTrain,
	}
	if len(segment.SubSegments) > 0 {
		features.Subsegments = &apipb.FeaturesSubSegments{
			Arrival: &apipb.SubSegmentsArrival{
				Min: segment.MinArrival,
				Max: segment.MaxArrival,
			},
		}
	}
	if segment.Transport.Subtype != (commonModels.TransportSubtype{}) {
		features.Subtype = &apipb.FeaturesSubtype{
			Id:    uint32(segment.Transport.Subtype.ID),
			Title: segment.Transport.Subtype.Title,
		}
	}
	if segment.Thread.DeluxeTrain != (commonModels.DeluxeTrain{}) {
		title := segment.Thread.DeluxeTrain.ShortTitle
		if title == "" {
			title = segment.Thread.DeluxeTrain.Title
		}
		features.NamedTrain = &apipb.FeaturesNamedTrain{
			Id:          uint32(segment.Thread.DeluxeTrain.ID),
			Title:       title,
			IsDeluxe:    segment.Thread.DeluxeTrain.IsDeluxe,
			IsHighSpeed: segment.Thread.DeluxeTrain.IsHighSpeed,
		}
	}
	return &features
}

func (c *Converter) tariffsToProto(tariffs *commonModels.SegmentTariffs) *apipb.MinTariffs {
	protoTariffs := &apipb.MinTariffs{
		Classes: &apipb.MinTariffsClasses{
			Platzkarte:  c.classPriceToProto(&tariffs.Classes.Platzkarte),
			Compartment: c.classPriceToProto(&tariffs.Classes.Compartment),
			Suite:       c.classPriceToProto(&tariffs.Classes.Suite),
			Common:      c.classPriceToProto(&tariffs.Classes.Common),
			Sitting:     c.classPriceToProto(&tariffs.Classes.Sitting),
			Soft:        c.classPriceToProto(&tariffs.Classes.Soft),
		},
	}
	if brokenClassesCode, ok := extractBrokenClassesCode(&tariffs.BrokenClasses); ok {
		protoTariffs.OptionalBrokenClassesCode = &apipb.MinTariffs_BrokenClassesCode{BrokenClassesCode: brokenClassesCode}
	}
	return protoTariffs
}

func (c *Converter) createPlacesDetails(classPrice *commonModels.ClassPrice) *apipb.PlacesDetails {
	result := apipb.PlacesDetails{}
	if classPrice.LowerSeats != nil {
		result.Lower = &apipb.PlaceDetails{
			Quantity: uint32(*classPrice.LowerSeats),
		}
	}
	if classPrice.UpperSeats != nil {
		result.Upper = &apipb.PlaceDetails{
			Quantity: uint32(*classPrice.UpperSeats),
		}
	}
	if classPrice.LowerSideSeats != nil {
		result.LowerSide = &apipb.PlaceDetails{
			Quantity: uint32(*classPrice.LowerSideSeats),
		}
	}
	if classPrice.UpperSideSeats != nil {
		result.UpperSide = &apipb.PlaceDetails{
			Quantity: uint32(*classPrice.UpperSideSeats),
		}
	}
	if proto.Equal(&result, &apipb.PlacesDetails{}) {
		return nil
	}
	return &result
}

func extractBrokenClassesCode(brokenClasses *commonModels.BrokenClasses) (apipb.BrokenClassesCode, bool) {
	hasOtherCode := false
	for _, codes := range [][]int{
		brokenClasses.Common,
		brokenClasses.Compartment,
		brokenClasses.Platzkarte,
		brokenClasses.Sitting,
		brokenClasses.Soft,
		brokenClasses.Suite,
		brokenClasses.Unknown,
	} {
		for _, code := range codes {
			if code == commonModels.TariffErrorSoldOut {
				return apipb.BrokenClassesCode_BROKEN_CLASSES_CODE_SOLD_OUT, true
			}
			hasOtherCode = true
		}
	}
	if hasOtherCode {
		return apipb.BrokenClassesCode_BROKEN_CLASSES_CODE_OTHER, true
	}
	return apipb.BrokenClassesCode_BROKEN_CLASSES_CODE_INVALID, false
}

func orderURLToProto(orderURL models.TransferVariantOrderURL) *apipb.OrderUrl {
	owner := apipb.OrderOwner_ORDER_OWNER_UNKNOWN
	switch orderURL.Owner {
	case models.OrderOwnerTrains:
		owner = apipb.OrderOwner_ORDER_OWNER_TRAINS
	case models.OrderOwnerUFS:
		owner = apipb.OrderOwner_ORDER_OWNER_UFS
	}

	return &apipb.OrderUrl{
		Owner: owner,
		Url:   orderURL.URL,
	}
}

func nearestDatesDirectionToProto(nearestDatesDirection models.NearestDatesDirection) *apipb.NearestDatesDirection {

	nearestDatesDirectionProto := &apipb.NearestDatesDirection{}
	switch nearestDatesDirection.Reason {
	case models.NearestDatesReasonUnknown:
		return nil
	case models.NearestDatesReasonNoRequestedDateDirectTrains:
		nearestDatesDirectionProto.Reason = apipb.NearestDatesReason_NEAREST_DATES_REASON_NO_REQUESTED_DATE_DIRECT_TRAINS
		nearestDatesDirectionProto.Dates = make([]*apipb.NearestDate, len(nearestDatesDirection.Dates))
		for i, date := range nearestDatesDirection.Dates {
			nearestDatesDirectionProto.Dates[i] = &apipb.NearestDate{
				Date: date.Date,
			}
		}
	case models.NearestDatesReasonNoDirectTrains:
		nearestDatesDirectionProto.Reason = apipb.NearestDatesReason_NEAREST_DATES_REASON_NO_DIRECT_TRAINS
	}
	return nearestDatesDirectionProto
}

func nearestDatesToProto(nearestDates models.NearestDates) *apipb.NearestDates {
	return &apipb.NearestDates{
		Forward:  nearestDatesDirectionToProto(nearestDates.Forward),
		Backward: nearestDatesDirectionToProto(nearestDates.Backward),
	}
}
