package handler

import (
	"context"
	"errors"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	httpclient "a.yandex-team.ru/travel/library/go/httputil/client"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"github.com/go-chi/chi/v5/middleware"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/proto"

	apipb "a.yandex-team.ru/travel/trains/search_api/api"
	pcpb "a.yandex-team.ru/travel/trains/search_api/api/price_calendar"
	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/pricecalendar"
	"a.yandex-team.ru/travel/trains/search_api/internal/searcher/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo"
)

type GRPCHandler struct {
	apipb.UnimplementedSearchApiServiceV1Server

	app                App
	logger             log.Logger
	converter          *Converter
	unifiedAgentClient unifiedagent.Client
}

func NewGRPCHandler(app App, unifiedAgentClient unifiedagent.Client, logger log.Logger) *GRPCHandler {
	return &GRPCHandler{
		app:                app,
		logger:             logger,
		converter:          NewConverter(logger),
		unifiedAgentClient: unifiedAgentClient,
	}
}

func (h *GRPCHandler) Search(ctx context.Context, request *apipb.Request) (*apipb.Response, error) {
	const funcName = "GRPCHandler.Search"

	if request.PointFrom == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument PointFrom is empty")
	}
	if request.PointTo == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument PointTo is empty")
	}
	if request.When == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument When is empty")
	}

	if request.RequestId != "" {
		ctx = context.WithValue(ctx, middleware.RequestIDKey, request.RequestId)
	}
	userArgs := models.UserArgs{
		IsBot:           request.IsBot || request.UserArgs.GetIsBot(),
		UaasExperiments: request.UserArgs.GetUaasExperiments(),
		Icookie:         request.UserArgs.GetIcookie(),
		UserDevice:      request.UserArgs.GetUserDevice(),
		YandexUID:       request.UserArgs.GetYandexUid(),
	}
	testContext := models.TestContext{
		MockImAuto: request.TestContext.GetMockImAuto(),
		MockImPath: request.TestContext.GetMockImPath(),
	}

	intResp, err := h.app.Search(
		ctx,
		request.PointFrom, request.PointTo,
		request.When, request.ReturnWhen,
		request.PinForwardSegmentId, request.PinBackwardSegmentId,
		request.OnlyDirect, request.OnlyOwnedPrices,
		userArgs,
		testContext,
	)

	if err != nil {
		var badResponseCodeError *httpclient.BadResponseCodeError
		if errors.As(err, &badResponseCodeError) && badResponseCodeError.Code == 400 {
			return nil, status.Errorf(codes.InvalidArgument, "bad search request: %v", err.Error())
		}
		return nil, status.Errorf(codes.Unavailable, "search error: %v", err.Error())
	}

	searchResult := h.converter.SearchResultToProto(intResp)

	if searchResult.Status == apipb.ResponseStatus_STATUS_DONE {
		searchLogRecord := &apipb.SearchLogRecord{
			Timestamp: time.Now().Unix(),
			Request:   proto.Clone(request).(*apipb.Request),
			Response:  proto.Clone(searchResult).(*apipb.Response),
		}
		go func() {
			err := h.unifiedAgentClient.Send(searchLogRecord, nil)
			if err != nil {
				h.logger.Errorf("%s: can not send log to unified_agent: %v", funcName, err)
			}
		}()
	}

	return searchResult, nil
}

func (h *GRPCHandler) SeoDirection(ctx context.Context, request *seopb.SeoDirectionRequest) (*seopb.SeoDirectionResponse, error) {
	const funcName = "GRPCHandler.SeoDirection"

	if request.FromSlug == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument FromSlug is empty")
	}
	if request.ToSlug == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument ToSlug is empty")
	}

	res, err := h.app.SeoDirection(ctx, request.FromSlug, request.ToSlug, lang.Ru)
	if err != nil {
		var notFoundError *seo.PointNotFoundError
		if errors.As(err, &notFoundError) {
			return nil, status.Error(codes.NotFound, notFoundError.Error())
		}
		return nil, err
	}
	return res, nil
}

func (h *GRPCHandler) PriceCalendar(ctx context.Context, request *pcpb.PriceCalendarRequest) (*pcpb.PriceCalendarResponse, error) {
	const funcName = "GRPCHandler.PriceCalendar"

	if request.PointFrom == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument PointFrom is empty")
	}
	if request.PointTo == "" {
		return nil, status.Error(codes.InvalidArgument, "required argument PointTo is empty")
	}

	userArgs := models.UserArgs{
		IsBot:           request.UserArgs.IsBot,
		UaasExperiments: request.UserArgs.UaasExperiments,
		Icookie:         request.UserArgs.Icookie,
		UserDevice:      request.UserArgs.UserDevice,
		YandexUID:       request.UserArgs.YandexUid,
	}
	res, err := h.app.PriceCalendar(ctx, request.PointFrom, request.PointTo, userArgs)
	if err != nil {
		var notFoundError *pricecalendar.PointNotFoundError
		if errors.As(err, &notFoundError) {
			return nil, status.Error(codes.NotFound, notFoundError.Error())
		}
		return nil, err
	}
	return res, nil
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		apipb.RegisterSearchApiServiceV1Server(server, h)
	}
}
