package handler

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/httputil"
	tariffmodels "a.yandex-team.ru/travel/trains/library/go/tariffs/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/query"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/errors"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
)

type HTTPHandler struct {
	app    App
	logger log.Logger
}

func NewHTTPHandler(app App, l log.Logger) *HTTPHandler {
	return &HTTPHandler{app: app, logger: l}
}

func (h *HTTPHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/searcher/api/direction/", h.GetDirection)
		r.Get("/searcher/api/open_direction/", h.GetOpenDirection)
		r.Get("/searcher/public-api/direction/", h.GetDirection)
		r.Get("/searcher/public-api/open_direction/", h.GetOpenDirection)

		r.Post("/indexer/api/direction/", h.Index)
		r.Post("/indexer/public-api/direction/", h.Index)
	}
}

const (
	departurePointExpressKey = "departure_point_express_id"
	arrivalPointExpressKey   = "arrival_point_express_id"
	departureDateKey         = "departure_date"
)

func (h *HTTPHandler) Index(w http.ResponseWriter, r *http.Request) {
	const funcName = "trains.internal.handlers.HTTPHandler.Index"

	span, ctx := opentracing.StartSpanFromContext(r.Context(), funcName)
	defer span.Finish()

	var params, err = url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		h.handleBadRequest(w, r, err)
		return
	}

	var (
		departurePointExpressID int
		arrivalPointExpressID   int
		departureDate           time.Time
		tariffInfos             = make([]*tariffmodels.DirectionTariffTrain, 0)
	)
	var parseErrorCauses []string

	parse := func(key string, paramParser func(value string) error) {
		value := params.Get(key)
		if err := paramParser(value); err != nil {
			parseErrorCauses = append(parseErrorCauses,
				fmt.Sprintf("bad value (%s) for required argument '%s'", value, key))
		}
	}

	parse(departurePointExpressKey, func(value string) (err error) {
		departurePointExpressID, err = strconv.Atoi(value)
		return err
	})

	parse(arrivalPointExpressKey, func(value string) (err error) {
		arrivalPointExpressID, err = strconv.Atoi(value)
		return err
	})

	parse(departureDateKey, func(value string) (err error) {
		departureDate, err = time.Parse(date.DateISOFormat, value)
		return err
	})

	if err = json.NewDecoder(r.Body).Decode(&tariffInfos); err != nil {
		parseErrorCauses = append(parseErrorCauses,
			fmt.Sprintf("body unmarshalling failed: %s", err))
	}

	if len(parseErrorCauses) > 0 {
		cause := strings.Join(parseErrorCauses, "; ")
		err = fmt.Errorf("%s: %s", funcName, cause)
		h.handleBadRequest(w, r, err)
		return
	}

	if err = h.app.Index(ctx, departurePointExpressID, arrivalPointExpressID, departureDate, tariffInfos); err != nil {
		if xerrors.Is(err, errors.ErrUnknownValue) {
			h.handleBadRequest(w, r, fmt.Errorf("%s: invalid tariff: %w", funcName, err))
			return
		}

		httputil.HandleError(
			fmt.Errorf("%s: tariff indexing failed: %w", funcName, err),
			http.StatusInternalServerError, w,
		)
		return
	}
	w.WriteHeader(http.StatusOK)
	_, err = w.Write([]byte("ok"))
	if err != nil {
		h.logger.Error("an error occurred while writing the response", log.Error(err))
	}
}

func (h *HTTPHandler) GetDirection(w http.ResponseWriter, r *http.Request) {
	const funcName = "trains.internal.handlers.HTTPHandler.GetDirection"

	span, ctx := opentracing.StartSpanFromContext(r.Context(), funcName)
	defer span.Finish()

	var params, err = url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		h.handleBadRequest(w, r, err)
		return
	}
	traceHTTPQuery(params, span)
	rawDirectionQuery := &query.RawDirectionQuery{
		DeparturePointKey:        getLastParam(params, "departure_point_key"),
		DepartureSettlementGeoID: getLastParam(params, "departure_settlement_geoid"),
		ArrivalPointKey:          getLastParam(params, "arrival_point_key"),
		ArrivalSettlementGeoID:   getLastParam(params, "arrival_settlement_geoid"),
		DepartureDate:            getLastParam(params, "departure_date"),
		MainReqID:                helpers.OptString(params.Get("main_reqid")),
		Language:                 getLastParam(params, "language"),
		OrderBy:                  getLastParam(params, "order_by"),
		TLD:                      getLastParam(params, "tld"),
		ExpFlags:                 getLastParam(params, "exp_flags"),
	}

	rawDirectionQuery.BrandFilter = parseFilters(params, "brand")
	rawDirectionQuery.CoachTypeFilter = parseFilters(params, "coach_type")

	response, err := h.app.Direction(ctx, rawDirectionQuery)
	h.writeResponse(w, err, response, false)
}

func (h *HTTPHandler) GetOpenDirection(w http.ResponseWriter, r *http.Request) {
	const funcName = "trains.internal.handlers.HTTPHandler.GetOpenDirection"

	span, ctx := opentracing.StartSpanFromContext(r.Context(), funcName)
	defer span.Finish()

	var params, err = url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		h.handleBadRequest(w, r, err)
		return
	}
	traceHTTPQuery(params, span)

	rawDirectionQuery := &query.RawDirectionQuery{
		DeparturePointKey:        getLastParam(params, "departure_point_key"),
		DepartureSettlementGeoID: getLastParam(params, "departure_settlement_geoid"),
		ArrivalPointKey:          getLastParam(params, "arrival_point_key"),
		ArrivalSettlementGeoID:   getLastParam(params, "arrival_settlement_geoid"),
		DepartureDate:            getLastParam(params, "departure_date"),
		MainReqID:                helpers.OptString(params.Get("main_reqid")),
		Language:                 getLastParam(params, "language"),
		OrderBy:                  getLastParam(params, "order_by"),
		TLD:                      getLastParam(params, "tld"),
		ExpFlags:                 getLastParam(params, "exp_flags"),
	}

	rawDirectionQuery.BrandFilter = parseFilters(params, "brand")
	rawDirectionQuery.CoachTypeFilter = parseFilters(params, "coach_type")

	response, err := h.app.OpenDirection(ctx, rawDirectionQuery)
	h.writeResponse(w, err, response, true)
}

func (h *HTTPHandler) writeResponse(w http.ResponseWriter, err error, response models.Response, emptyStatusCode bool) {
	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		httputil.HandleError(fmt.Errorf("internal error"), http.StatusInternalServerError, w)
		return
	}

	responseBody, err := json.Marshal(response)
	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		httputil.HandleError(fmt.Errorf("internal error"), http.StatusInternalServerError, w)
		return
	}

	if response.IsError() {
		w.WriteHeader(http.StatusBadRequest)
		_, err = w.Write(responseBody)
		if err != nil {
			h.logger.Error("an error occurred while writing the response", log.Error(err))
		}
		return
	}

	if response.IsEmpty() && emptyStatusCode {
		w.WriteHeader(http.StatusNoContent)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)
	_, err = w.Write(responseBody)
	if err != nil {
		h.logger.Error("an error occurred while writing the response", log.Error(err))
	}
}

func (h HTTPHandler) handleBadRequest(w http.ResponseWriter, r *http.Request, err error) {
	httputil.HandleError(err, http.StatusBadRequest, w)

	route := r.URL.Path
	h.logger.Info(
		fmt.Sprintf("bad request on route %s", route),
		log.Error(err),
		log.String("route", route),
		log.String("query", r.URL.RawQuery),
		log.String("method", r.Method),
	)
}

// charon_proxy иногда отдает departure_settlement_geoid, и нас интересует последний.
func getLastParam(params url.Values, paramName string) string {
	if params == nil {
		return ""
	}
	values := params[paramName]
	valuesLength := len(values)
	if valuesLength > 0 {
		return values[valuesLength-1]
	}
	return ""
}

func traceHTTPQuery(params url.Values, span opentracing.Span) {
	for k, v := range params {
		span.SetTag("query."+k, strings.Join(v, ","))
	}
}

func parseFilters(params url.Values, paramName string) (result []string) {
	if filters, found := params[paramName]; found {
		for _, filter := range filters {
			if len(filter) > 0 {
				result = append(result, filter)
			}
		}
	}
	return result
}
