package indexer

import (
	"context"
	"fmt"
	"time"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/logbroker"
	tpb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/trains/library/go/tariffs/models"
	api "a.yandex-team.ru/travel/trains/search_api/api/tariffs"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/errors"
)

type Indexer struct {
	logger         log.Logger
	tariffProducer *logbroker.Producer
	metrics        *indexerMetrics
}

func New(
	logger log.Logger,
	tariffProducer *logbroker.Producer,
) *Indexer {
	return &Indexer{
		logger:         logger,
		tariffProducer: tariffProducer,
		metrics:        newIndexerMetrics(),
	}
}

func (i *Indexer) Index(
	ctx context.Context,
	departurePointExpressID int,
	arrivalPointExpressID int,
	departureDate time.Time,
	trains []*models.DirectionTariffTrain,
) error {
	const funcName = "trains.internal.indexer.Indexer.Index"

	span, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer span.Finish()

	message := new(api.DirectionTariffInfo)
	message.DeparturePointExpressId = int32(departurePointExpressID)
	message.ArrivalPointExpressId = int32(arrivalPointExpressID)
	message.DepartureDate = date.GetProtoFromDate(departureDate)

	nowTS := timestamppb.Now()
	message.CreatedAt = nowTS
	message.UpdatedAt = nowTS

	var err error
	if message.Data, err = models.MapDirectionTariffTrainJSONToAPI(ctx, trains...); err != nil {
		return fmt.Errorf("%s: mapping json tariff failed: %w", funcName, err)
	}

	for _, train := range message.Data {
		if err := i.validateTrain(train); err != nil {
			i.metrics.parsingTariffErrors.Inc()
			return fmt.Errorf("%s: invalid train: %w", funcName, err)
		}
	}

	if err = i.tariffProducer.Write(message); err != nil {
		return fmt.Errorf("%s: tariff writing failed: %w", funcName, err)
	}
	return nil
}

func (i *Indexer) validateTrain(train *api.DirectionTariffTrain) error {
	for _, place := range train.Places {
		if err := i.validatePrice(place.Price); err != nil {
			return fmt.Errorf("bad price: %w", err)
		}
		if err := i.validatePriceDetails(place.PriceDetails); err != nil {
			return fmt.Errorf("bad price details: %w", err)
		}
	}
	return nil
}

func (i *Indexer) validatePriceDetails(details *api.TrainPlacePriceDetails) error {
	if details != nil {
		for cause, price := range map[string]*tpb.TPrice{
			"ticket_price":  details.TicketPrice,
			"service_price": details.ServicePrice,
			"fee":           details.Fee,
		} {
			if err := i.validatePrice(price); err != nil {
				return fmt.Errorf("bad %s: %w", cause, err)
			}
		}
	}
	return nil
}

func (i *Indexer) validatePrice(price *tpb.TPrice) error {
	if price != nil && price.Currency == tpb.ECurrency_C_UNKNOWN {
		return fmt.Errorf("unknown currency: %w", errors.ErrUnknownValue)
	}
	return nil
}
