package runmask

import (
	"math"
	"time"
)

const (
	EmptyBucket = int32(0)
	FullBucket  = int32(math.MaxInt32)
)

type RunMask struct {
	mask [12]int32
}

func New(rawMask []int32) *RunMask {
	mask := new(RunMask)
	for i, bucket := range rawMask {
		if i >= 12 {
			break
		}
		mask.mask[i] = bucket
	}
	return mask
}

func NewEmpty() *RunMask {
	return new(RunMask)
}

func NewFullMask() *RunMask {
	m := new(RunMask)
	for i := range m.mask {
		m.mask[i] = FullBucket
	}
	return m
}

func (m *RunMask) SetByDate(date time.Time) {
	bucketNo, bucket := bucketByDate(date)
	m.mask[bucketNo] = m.mask[bucketNo] | bucket
}

func (m *RunMask) RunsAt(date time.Time) bool {
	bucketNo, bucket := bucketByDate(date)
	maskBucket := m.mask[bucketNo]
	return (maskBucket & bucket) == bucket
}

func (m *RunMask) RawMask() []int32 {
	return m.mask[:]
}

func MaskByDate(date time.Time) *RunMask {
	mask := new(RunMask)
	mask.SetByDate(date)
	return mask
}

func IsEmptyYearDays(mask []int32) bool {
	for _, bucket := range mask {
		if bucket != 0 {
			return false
		}
	}
	return true
}

func bucketByDate(date time.Time) (int, int32) {
	// every month stored isolated in a separate bucket.
	bucket := int32(1) << (date.Day() - 1)
	return int(date.Month() - 1), bucket
}
