package registry

import (
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/updater"
)

type (
	DictLoadFn func(*resourcestorage.Loader) error

	DictLoader struct {
		logger         log.Logger
		resourceReader resourcestorage.StorageReader
		registry       *RepositoryRegistry

		loadFn          DictLoadFn
		resourceKey     string
		sample          proto.Message
		lastLoadingTime time.Time
	}
)

func NewDictLoader(
	logger log.Logger,
	resourceReader resourcestorage.StorageReader,
	loadFn DictLoadFn,
	resourceKey string,
	sample proto.Message,
) *DictLoader {
	return &DictLoader{
		logger:         logger,
		resourceReader: resourceReader,
		loadFn:         loadFn,
		resourceKey:    resourceKey,
		sample:         sample,
	}
}

func (l *DictLoader) Load() (err error) {
	defer errutil.Wrap(&err, "DictLoader.Load() %s", l.resourceKey)

	resourceStoreTime, err := l.resourceReader.GetTimestamp(l.resourceKey)
	if err != nil {
		return err
	}

	if !l.lastLoadingTime.Before(resourceStoreTime) {
		return updater.AlreadyUpdated
	}

	resourceLoader := resourcestorage.NewLoader(l.sample, l.resourceKey, l.resourceReader, l.logger)
	err = l.loadFn(resourceLoader)
	if err != nil {
		return err
	}
	l.lastLoadingTime = resourceStoreTime
	return
}
