package express

import (
	"sync"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/traincity"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/updater"
)

type Repository struct {
	logger              log.Structured
	registry            *registry.RepositoryRegistry
	trainCityRepository *traincity.Repository

	updateMutex sync.Mutex
	cacheValue  atomic.Value
	validCache  bool
}

func NewRepository(logger log.Logger, repoRegistry *registry.RepositoryRegistry, trainCityRepository *traincity.Repository) *Repository {
	repo := &Repository{
		logger:              logger.Structured(),
		registry:            repoRegistry,
		trainCityRepository: trainCityRepository,
	}

	repoRegistry.AddRepositoryObserver(registry.StationCode, repo)
	return repo
}

func (r *Repository) OnDictUpdate(_ registry.DictType) {
	r.validCache = false
}

func (r *Repository) UpdateCache() error {
	r.updateMutex.Lock()
	defer r.updateMutex.Unlock()

	if r.validCache {
		return updater.AlreadyUpdated
	}

	cache := NewCache(r.logger, r.registry, r.trainCityRepository).Build()
	r.cacheValue.Store(cache)
	r.validCache = true
	return nil
}

func (r *Repository) FindExpressID(point points.Point) int {
	cache := r.cacheValue.Load().(*Cache)

	switch point.Type() {
	case points.SettlementType:
		if station := r.trainCityRepository.Get(point.ID()); station != nil {
			return cache.stationIDToExpressID[station.Id]
		}
	case points.StationType:
		return cache.stationIDToExpressID[point.ID()]
	}
	return 0
}
