package helpers

import (
	"fmt"
	"math"

	tpb "a.yandex-team.ru/travel/proto"
	commonModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/common/models"
)

const (
	pricePrecision = 2
)

var (
	priceCurrencyMap = map[string]tpb.ECurrency{
		"RUB": tpb.ECurrency_C_RUB,
	}
	currencySymbols = map[tpb.ECurrency]string{
		tpb.ECurrency_C_RUB: "₽",
		tpb.ECurrency_C_USD: "$",
		tpb.ECurrency_C_EUR: "€",
		tpb.ECurrency_C_JPY: "¥",
	}
)

func GetCurrencySymbol(c tpb.ECurrency) string {
	return currencySymbols[c]
}

func PrettyIntegerPrice(p *tpb.TPrice) string {
	return fmt.Sprintf("%d %s", int(GetFloatValue(p)), GetCurrencySymbol(p.Currency))
}

func GetFloatValue(p *tpb.TPrice) float64 {
	return float64(p.Amount) / math.Pow10(int(p.Precision))
}

func GetMinPrice(lhs *tpb.TPrice, rhs *tpb.TPrice) *tpb.TPrice {
	if lhs == nil {
		return rhs
	}
	if rhs == nil {
		return lhs
	}

	if GetFloatValue(lhs) < GetFloatValue(rhs) {
		return lhs
	}
	return rhs
}

func PriceIsEqualOrNil(lhs *tpb.TPrice, rhs *tpb.TPrice) bool {
	if lhs == nil && rhs == nil {
		return true
	} else if lhs == nil || rhs == nil {
		return false
	}

	lValue := GetFloatValue(lhs)
	rValue := GetFloatValue(rhs)
	return lValue == rValue && lhs.Currency == rhs.Currency
}

func PriceToProto(price *commonModels.Price) (*tpb.TPrice, error) {
	funcName := "PriceToProto"

	if price == nil {
		return nil, nil
	}

	priceCurrency, ok := priceCurrencyMap[price.Currency]
	if !ok {
		return nil, fmt.Errorf("%s: can not convert price due to not supported currency=%s", funcName, price.Currency)
	}

	return &tpb.TPrice{
		Currency:  priceCurrency,
		Precision: pricePrecision,
		Amount:    int64(float64(price.Value) * math.Pow10(pricePrecision)),
	}, nil
}

func Sum(lhs *tpb.TPrice, rhs *tpb.TPrice) (*tpb.TPrice, error) {
	funcName := "Sum"
	if lhs == nil || rhs == nil {
		return nil, fmt.Errorf("%s: nil argument", funcName)
	}

	if lhs.Currency != rhs.Currency {
		return nil, fmt.Errorf("%s: cannot sum prices with different currency", funcName)
	}
	if lhs.Precision < rhs.Precision {
		// меняем местами так чтобы Precision был больше в lhs
		a := lhs
		lhs = rhs
		rhs = a
	}
	precisionDiff := lhs.Precision - rhs.Precision
	rhsAmount := rhs.Amount
	if precisionDiff != 0 {
		rhsAmount = rhs.Amount * int64(math.Pow(10, float64(precisionDiff)))
	}
	result := tpb.TPrice{
		Currency:  lhs.Currency,
		Precision: lhs.Precision,
		Amount:    rhsAmount + lhs.Amount,
	}

	return &result, nil
}
