package i18n

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/geobase"

	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/geo"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

var LinguisticFormNotFoundError = xerrors.New("linguistic form not found")
var UnexpectedGrammaticalCaseError = xerrors.New("unexpected grammatical case")

type LinguisticsTranslator struct {
	geoBase               geo.Geobase
	keyset                Keyset
	languageFallbacks     map[lang.Lang]lang.Lang
	languageCaseFallbacks map[lang.LinguisticForm]lang.LinguisticForm
}

func NewLinguisticsTranslator(
	geoBase geo.Geobase,
	keyset Keyset,
	languageFallbacks map[lang.Lang]lang.Lang,
	languageCaseFallbacks map[lang.LinguisticForm]lang.LinguisticForm,
) *LinguisticsTranslator {
	return &LinguisticsTranslator{
		geoBase:               geoBase,
		keyset:                keyset,
		languageFallbacks:     languageFallbacks,
		languageCaseFallbacks: languageCaseFallbacks,
	}
}

func (t *LinguisticsTranslator) FindPreposition(language lang.Lang, preposition lang.Preposition, chain []lang.Prepositions) (string, error) {
	langCasePair := lang.NewPrepositionForm(language, preposition)
	for _, prepositions := range chain {
		if result, found := prepositions[langCasePair]; found {
			return result, nil
		}
	}
	return t.keyset.ExecuteSingular(preposition.String(), language)
}

func (t *LinguisticsTranslator) FindLinguisticForm(language lang.Lang, grammaticalCase lang.GrammaticalCase, linguisticsChain []lang.Linguistics) (string, error) {
	findLinguisticForm := func(langCasePair lang.LinguisticForm) (string, bool) {
		for _, linguistics := range linguisticsChain {
			if linguisticForm, found := linguistics[langCasePair]; found {
				return linguisticForm, true
			}
		}
		return "", false
	}

	cases := []lang.GrammaticalCase{grammaticalCase}
	if grammaticalCase != lang.Nominative {
		cases = append(cases, lang.Nominative)
	}

	for _, grammaticalCase := range cases {
		if form, found := findLinguisticForm(lang.NewLinguisticForm(language, grammaticalCase)); found {
			return form, nil
		}

		if fallback, found := t.languageCaseFallbacks[lang.NewLinguisticForm(language, grammaticalCase)]; found {
			if form, found := findLinguisticForm(fallback); found {
				return form, nil
			}
		}
	}

	if langFallback, found := t.languageFallbacks[language]; found {
		if form, found := findLinguisticForm(lang.NewLinguisticForm(langFallback, lang.Nominative)); found {
			return form, nil
		}
	}
	return "", LinguisticFormNotFoundError
}

func (t *LinguisticsTranslator) FindLinguisticFormByGeoBase(geoID int32, language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	const funcName = "pkg.i18n.LinguisticsTranslator.FindLinguisticFormByGeoBase"

	linguistic, err := t.geoBase.GetLinguistics(geobase.ID(geoID), string(language))
	if err != nil {
		return "", fmt.Errorf("%s fails: %w", funcName, err)
	}

	translation, err := getLinguisticByCase(linguistic, grammaticalCase)
	if err != nil {
		return "", fmt.Errorf("%s fails: %w", funcName, err)
	}
	return translation, nil
}

func getLinguisticByCase(linguistics *geobase.Linguistics, grammaticalCase lang.GrammaticalCase) (string, error) {
	if linguistics == nil {
		return "", LinguisticFormNotFoundError
	}
	switch grammaticalCase {
	case lang.Nominative:
		return linguistics.NominativeCase, nil
	case lang.Genitive:
		return linguistics.GenitiveCase, nil
	case lang.Accusative:
		return linguistics.AccusativeCase, nil
	case lang.Locative:
		return linguistics.LocativeCase, nil
	case lang.Dative:
		return linguistics.DativeCase, nil

	default:
		return "", UnexpectedGrammaticalCaseError
	}
}
