package i18n

import (
	"fmt"

	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type NamedTrain struct {
	value                 *rasp.TNamedTrain
	linguisticsTranslator *LinguisticsTranslator
	keyset                Keyset
}

func NewNamedTrain(value *rasp.TNamedTrain, linguisticsTranslator *LinguisticsTranslator, keyset Keyset) NamedTrain {
	return NamedTrain{
		value:                 value,
		linguisticsTranslator: linguisticsTranslator,
		keyset:                keyset,
	}
}

func (s NamedTrain) Title(language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	title, err := s.linguisticsTranslator.FindLinguisticForm(language, grammaticalCase, []lang.Linguistics{s.getTitleTranslations()})
	if err != nil {
		return "", fmt.Errorf("%s fails: %w", "pkg.translations.models.NamedTrain.Title", err)
	}
	return title, nil
}

func (s NamedTrain) ShortTitle(language lang.Lang) (string, error) {
	const funcName = "pkg.translations.models.NamedTrain.ShortTitle"

	title, _ := s.Title(language, lang.Nominative)
	if len(title) > 0 && s.value.IsDeluxe {
		text, err := s.keyset.ExecuteSingular("deluxe_train_with_title", language, map[string]interface{}{
			"title": title,
		})
		if err != nil {
			return "", fmt.Errorf("%s fails: %w", funcName, err)
		}
		return text, nil
	} else if s.value.IsDeluxe {
		text, err := s.keyset.ExecuteSingular("deluxe_train", language)
		if err != nil {
			return "", fmt.Errorf("%s fails: %w", funcName, err)
		}
		return text, nil
	} else if len(title) > 0 {
		return fmt.Sprintf("«%s»", title), nil
	}
	return "", nil
}

func (s NamedTrain) getTitleTranslations() lang.Linguistics {
	if s.value.Title == nil {
		return make(lang.Linguistics)
	}

	linguistics := lang.Linguistics{
		lang.RuNominative: s.value.Title.Ru,
		lang.EnNominative: s.value.Title.En,
		lang.TrNominative: s.value.Title.Tr,
		lang.UkNominative: s.value.Title.Uk,
	}

	for key, value := range linguistics {
		if value == "" {
			delete(linguistics, key)
		}
	}
	return linguistics
}
