package i18n

import (
	"fmt"

	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type Settlement struct {
	value                 *rasp.TSettlement
	linguisticsTranslator *LinguisticsTranslator
}

func NewSettlement(value *rasp.TSettlement, linguisticsTranslator *LinguisticsTranslator) Settlement {
	return Settlement{
		value:                 value,
		linguisticsTranslator: linguisticsTranslator,
	}
}

func (s Settlement) Preposition(language lang.Lang, preposition lang.Preposition) (string, error) {
	prepositions := lang.Prepositions{
		lang.RuPrepositionTo: s.value.TitleRuPreposition,
		lang.RuPrepositionIn: s.value.TitleRuPreposition,
	}
	pp, err := s.linguisticsTranslator.FindPreposition(language, preposition, []lang.Prepositions{prepositions})
	if err != nil {
		return "", fmt.Errorf("%s fails: %w", "pkg.translations.models.Settlement.PrepositionTo", err)
	}
	return pp, nil
}

func (s Settlement) Title(language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	if title, err := s.linguisticsTranslator.FindLinguisticForm(language, grammaticalCase, []lang.Linguistics{s.getTitleTranslations()}); err == nil {
		return title, nil
	}

	title, err := s.linguisticsTranslator.FindLinguisticFormByGeoBase(s.value.GeoId, language, grammaticalCase)
	if err != nil {
		return "", fmt.Errorf("%s fails: %w", "pkg.translations.models.Settlement.Title", err)
	}
	return title, nil
}

func (s Settlement) PopularTitle(language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	return s.Title(language, grammaticalCase)
}

func (s Settlement) getTitleTranslations() lang.Linguistics {
	linguistics := make(lang.Linguistics)

	if s.value.Title == nil {
		return linguistics
	}

	if s.value.Title.Ru != nil {
		linguistics[lang.RuNominative] = s.value.Title.Ru.Nominative
		linguistics[lang.RuGenitive] = s.value.Title.Ru.Genitive
		linguistics[lang.RuAccusative] = s.value.Title.Ru.Accusative
		linguistics[lang.RuLocative] = s.value.Title.Ru.Prepositional
	}

	if s.value.Title.En != nil {
		linguistics[lang.EnNominative] = s.value.Title.En.Nominative
	}

	if s.value.Title.En != nil {
		linguistics[lang.TrNominative] = s.value.Title.Tr.Nominative
	}

	if s.value.Title.En != nil {
		linguistics[lang.UkNominative] = s.value.Title.Uk.Nominative
		linguistics[lang.UkAccusative] = s.value.Title.Uk.Nominative
	}

	for key, value := range linguistics {
		if value == "" {
			delete(linguistics, key)
		}
	}
	return linguistics
}
