package i18n

import (
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type monthForms map[string]string

type TimeTranslator struct {
	keyset map[string]monthForms
}

type TimeTranslatorOption func(factory *TimeTranslator) *TimeTranslator

func WithKeysetPath(logger log.Logger, cfg *Config) TimeTranslatorOption {
	return func(translator *TimeTranslator) *TimeTranslator {
		file, err := os.Open(cfg.KeysetMonthPath)
		if err != nil {
			logger.Fatal("json config not found",
				log.String("file_path", cfg.KeysetMonthPath),
				log.Error(err),
			)
		}
		decoder := json.NewDecoder(file)
		if err := decoder.Decode(&translator.keyset); err != nil {
			logger.Fatal("json config has unexpected format",
				log.String("file_path", cfg.KeysetMonthPath),
				log.Error(err),
			)
		}
		logger.Info("config loaded", log.String("file_path", cfg.KeysetMonthPath))
		return translator
	}
}

func WithFakeForms() TimeTranslatorOption {
	return func(translator *TimeTranslator) *TimeTranslator {
		for _, l := range []lang.Lang{lang.Uk, lang.Ru, lang.En, lang.Tr} {
			forms := make(monthForms)
			for i := 1; i <= 12; i++ {
				forms[fmt.Sprintf("g%d", i)] = time.Month(i).String()
			}
			translator.keyset[l.String()] = forms
		}
		return translator
	}
}

func NewTimeTranslator(options ...TimeTranslatorOption) *TimeTranslator {
	translator := &TimeTranslator{
		keyset: make(map[string]monthForms),
	}
	for _, option := range options {
		translator = option(translator)
	}
	return translator
}

func (t *TimeTranslator) FormatHumanDate(language lang.Lang, date time.Time) (string, error) {
	month, err := t.FormatMonth(language, date.Month())
	if err != nil {
		return "", err
	}

	if date.Year() == time.Now().Year() {
		return fmt.Sprintf("%d %s", date.Day(), month), nil
	} else {
		return fmt.Sprintf("%d %s %d", date.Day(), month, date.Year()), nil
	}
}

func (t *TimeTranslator) FormatMonth(language lang.Lang, month time.Month) (string, error) {
	forms, found := t.keyset[language.String()]
	if !found {
		return "", fmt.Errorf("month translation not found for language %s", language)
	}

	key := fmt.Sprintf("g%d", int(month))
	if form, found := forms[key]; found {
		if language == lang.Ru || language == lang.Uk {
			form = strings.ToLower(form)
		}
		return form, nil
	}

	return "", xerrors.New("month translation not found for genitive case")
}
