package limitcondition

import (
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
)

const notInTabloMajority = rasp.TStation_MAJORITY_NOT_IN_TABLO

type LimitCondition struct {
	DepartureMaxMajorityID rasp.TStation_EMajority
	ArrivalMaxMajorityID   rasp.TStation_EMajority
	transportTypes         []rasp.TTransport_EType
	extended               bool
}

func WithTransportTypes(transportTypes ...rasp.TTransport_EType) func(condition *LimitCondition) {
	return func(condition *LimitCondition) {
		condition.transportTypes = transportTypes
	}
}

func AsExtended() func(condition *LimitCondition) {
	return func(condition *LimitCondition) {
		condition.extended = true
	}
}

func New(departure, arrival points.Point, options ...func(condition *LimitCondition)) *LimitCondition {
	condition := &LimitCondition{
		DepartureMaxMajorityID: rasp.TStation_MAJORITY_STATION,
		ArrivalMaxMajorityID:   rasp.TStation_MAJORITY_STATION,
		transportTypes:         make([]rasp.TTransport_EType, 0),
	}
	for _, op := range options {
		op(condition)
	}

	isDifferentSettlements := false
	isStationsInTheSameCity := false
	isFromCityToStationInOtherCity := false
	isFromCityToStationInTheSameCity := false
	isFromStationInOtherCityToCity := false
	isFromStationInTheSameCityToCity := false

	if departure.Type() == points.SettlementType && arrival.Type() == points.SettlementType {
		isDifferentSettlements = true
	} else if departure.Type() == points.StationType && arrival.Type() == points.StationType {
		departureSettlementID := departure.SettlementID()
		arrivalSettlementID := arrival.SettlementID()
		if departureSettlementID == arrivalSettlementID && departureSettlementID != 0 {
			isStationsInTheSameCity = true
		}
	} else if departure.Type() == points.StationType {
		if departure.SettlementID() == arrival.ID() {
			isFromStationInTheSameCityToCity = true
		} else {
			isFromStationInOtherCityToCity = true
		}
	} else if arrival.Type() == points.StationType {
		if arrival.SettlementID() == departure.ID() {
			isFromCityToStationInTheSameCity = true
		} else {
			isFromCityToStationInOtherCity = true
		}
	}

	if isDifferentSettlements {
		condition.DepartureMaxMajorityID = notInTabloMajority
		condition.ArrivalMaxMajorityID = notInTabloMajority
	} else if isStationsInTheSameCity || isFromCityToStationInTheSameCity || isFromStationInTheSameCityToCity {
		if !condition.extended {
			condition.transportTypes = filterInternalTransportTypes(condition.transportTypes)
		}
	} else if isFromCityToStationInOtherCity {
		condition.DepartureMaxMajorityID = notInTabloMajority
	} else if isFromStationInOtherCityToCity {
		condition.ArrivalMaxMajorityID = notInTabloMajority
	}

	return condition
}

func (c *LimitCondition) IsEmptyTransportTypes() bool {
	return len(c.transportTypes) == 0
}

func filterInternalTransportTypes(transportTypes []rasp.TTransport_EType) []rasp.TTransport_EType {
	var newTransportTypes []rasp.TTransport_EType
	for _, tt := range transportTypes {
		if tt != rasp.TTransport_TYPE_TRAIN && tt != rasp.TTransport_TYPE_PLANE {
			newTransportTypes = append(newTransportTypes, tt)
		}
	}
	return newTransportTypes
}
