package logbroker

import (
	"os"
	"time"

	uzap "go.uber.org/zap"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/recipe"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/yt/go/yt"
)

const (
	defaultEndpoint      = "logbroker.yandex.net"
	defaultReconnectWait = 15 * time.Second
)

type ProducerConfig struct {
	OAuthToken string `config:"logbroker-producer-oauthtoken, required" yaml:"oauth-token"`
	Endpoint   string `config:"logbroker-producer-endpoint, required"`
	Topic      string `config:"logbroker-producer-topic, required"`
	SourceID   string `config:"logbroker-producer-sourceid, required" yaml:"source-id"`
	TestEnv    *recipe.Env
}

var DefaultProducerConfig = ProducerConfig{
	Endpoint: defaultEndpoint,
	SourceID: os.Getenv("DEPLOY_POD_ID"),
}

type ConsumerConfig struct {
	OAuthToken    string        `config:"logbroker-consumer-oauthtoken" yaml:"oauth-token"`
	Endpoint      string        `config:"logbroker-consumer-endpoint, required"`
	Topic         string        `config:"logbroker-consumer-topic, required"`
	Clusters      []string      `config:"logbroker-consumer-clusters, required"`
	SourceID      string        `config:"logbroker-consumer-sourceid, required" yaml:"source-id"`
	ReconnectWait time.Duration `config:"logbroker-consumer-reconnectwait, required" yaml:"reconnect-wait"`

	ConsumerPrefix string `config:"logbroker-consumer-consumer-prefix, required" yaml:"consumer-prefix"`
	YtLockPrefix   string `config:"logbroker-consumer-ytlock-prefix, required" yaml:"ytlock-prefix"`
	YtLockPoolSize uint   `config:"logbroker-consumer-ytlock-poolsize, required" yaml:"ytlock-pool-size"`

	TestEnv *recipe.Env
}

var DefaultConsumerConfig = ConsumerConfig{
	Endpoint:       defaultEndpoint,
	SourceID:       os.Getenv("DEPLOY_POD_ID"),
	ReconnectWait:  defaultReconnectWait,
	YtLockPoolSize: 1,
}

func loggerWithContext(logger log.Logger, topic string) log.Logger {
	return log.With(logger,
		log.String("topic", topic),
		log.String("module", "logbroker"),
	)
}

func NewProducer(cfg *ProducerConfig, logger log.Logger) (*logbroker.Producer, error) {
	if cfg.TestEnv != nil {
		return logbroker.NewProducerWithRecipe(cfg.TestEnv, "", cfg.Topic, logger), nil
	}

	return logbroker.NewProducer(
		cfg.Topic,
		cfg.Endpoint,
		cfg.SourceID,
		logbroker.NewOAuthCredentialsProvider(cfg.OAuthToken),
		withLevel(loggerWithContext(logger, cfg.Topic), log.InfoLevel),
	)
}

func NewConsumer(cfg *ConsumerConfig, readTimestamp time.Time, ytClient yt.Client, logger log.Logger) (*logbroker.Consumer, error) {
	if cfg.TestEnv != nil {
		return logbroker.NewConsumerWithRecipe(cfg.TestEnv, []string{""}, cfg.Topic, logger)
	}

	lbConsumer := cfg.ConsumerPrefix
	return logbroker.NewConsumerWithLbConsumerGenerating(
		cfg.Clusters,
		cfg.Endpoint,
		cfg.Topic,
		lbConsumer,
		readTimestamp,
		logbroker.NewOAuthCredentialsProvider(cfg.OAuthToken),
		withLevel(loggerWithContext(logger, cfg.Topic), log.InfoLevel),
		logbroker.NewYtLockGroup(ytClient, cfg.YtLockPrefix, cfg.YtLockPoolSize),
	)
}

func withLevel(logger log.Logger, level log.Level) log.Logger {
	zapLogger, ok := logger.(*zap.Logger)
	if !ok {
		return logger
	}

	return &zap.Logger{L: zapLogger.L.WithOptions(uzap.IncreaseLevel(zap.ZapifyLevel(level)))}
}
