package points

import (
	"fmt"
	"strconv"

	"a.yandex-team.ru/library/go/core/xerrors"
	dicts "a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
)

type RegionCapitalRepo interface {
	GetRegionCapital(geoID int) *dicts.TSettlement
}

type Parser struct {
	repoRegistry      *registry.RepositoryRegistry
	regionCapitalRepo RegionCapitalRepo
}

func NewParser(repoRegistry *registry.RepositoryRegistry, regionCapitalRepo RegionCapitalRepo) *Parser {
	return &Parser{
		repoRegistry:      repoRegistry,
		regionCapitalRepo: regionCapitalRepo,
	}
}

func (p *Parser) ParsePoint(pointKey, geoID string) (Point, error) {
	if len(pointKey) > 0 && len(geoID) > 0 {
		return nil, fmt.Errorf("one of arguments have to be specified: key=%s, geo_id=%s", pointKey, geoID)
	}

	pointByKey, err := p.ParseByPointKey(pointKey)
	if xerrors.Is(err, UnexpectedFormatErr) {
		return nil, fmt.Errorf("invalid point key: key=%s", pointKey)
	}
	if xerrors.Is(err, NotExistsErr) {
		return nil, fmt.Errorf("unknown point key: key=%s", pointKey)
	}
	if err != nil {
		return nil, fmt.Errorf("pkg.points.Parser.ParsePoint: %w", err)
	}

	if pointByKey != nil {
		return pointByKey, nil
	}

	pointByGeoID, err := p.ParseSettlementByGeoID(geoID)
	if xerrors.Is(err, UnexpectedFormatErr) {
		return nil, fmt.Errorf("invalid geo_id: geo_id=%s", geoID)
	}
	if xerrors.Is(err, NotExistsErr) {
		return nil, fmt.Errorf("unknown geo_id: geo_id=%s", geoID)
	}
	if err != nil {
		return nil, fmt.Errorf("pkg.points.Parser.ParsePoint: %w", err)
	}

	if pointByGeoID != nil {
		return pointByGeoID, nil
	}

	return nil, fmt.Errorf("one of arguments have to be specified: key=%s, geo_id=%s", pointKey, geoID)
}

func (p *Parser) ParseByPointKey(key string) (point Point, err error) {
	defer func() {
		if err != nil {
			err = xerrors.Errorf("ParseByPointKey(%s): %w", key, err)
		}
	}()

	if len(key) == 0 {
		return nil, nil
	}

	if len(key) == 1 {
		return nil, UnexpectedFormatErr
	}

	prefix := key[0:1]
	pointID, err := strconv.Atoi(key[1:])
	if err != nil {
		return nil, UnexpectedFormatErr
	}

	switch prefix {
	case "g":
		if point = p.getSettlementByGeoID(pointID); point != nil {
			return
		}
	case "c":
		settlementRepo := p.repoRegistry.GetSettlementRepo()
		if settlement, found := settlementRepo.Get(int32(pointID)); found {
			return NewSettlement(settlement), nil
		}
	case "s":
		stationRepo := p.repoRegistry.GetStationRepo()
		if station, found := stationRepo.Get(int32(pointID)); found {
			return NewStation(station), nil
		}
	default:
		return nil, UnexpectedFormatErr
	}

	return nil, NotExistsErr
}

func (p *Parser) ParseSettlementByGeoID(geoIDString string) (Point, error) {
	if len(geoIDString) == 0 {
		return nil, nil
	}

	geoID, err := strconv.Atoi(geoIDString)
	if err != nil {
		return nil, UnexpectedFormatErr
	}

	point := p.getSettlementByGeoID(geoID)
	if point != nil {
		return point, nil
	}
	settlement := p.regionCapitalRepo.GetRegionCapital(geoID)
	if settlement != nil {
		return NewSettlement(settlement), nil
	}
	return nil, NotExistsErr
}

func (p *Parser) getSettlementByGeoID(geoID int) Point {
	settlementRepo := p.repoRegistry.GetSettlementRepo()
	if settlement, found := settlementRepo.GetByGeoID(int32(geoID)); found && !settlement.IsHidden {
		return NewSettlement(settlement)
	}
	return nil
}

func (p *Parser) ParseBySlug(slug string) (point Point, err error) {
	defer func() {
		if err != nil {
			err = xerrors.Errorf("ParseBySlug(%s): %w", slug, err)
		}
	}()

	city, ok := p.repoRegistry.GetSettlementRepo().SettlementBySlug.GetBySlug(slug)
	if ok {
		return NewSettlement(city), nil
	}
	station, ok := p.repoRegistry.GetStationRepo().StationBySlug.GetBySlug(slug)
	if ok {
		return NewStation(station), nil
	}
	return nil, NotExistsErr
}
