package points

import (
	"strconv"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

type PointType int

const (
	SettlementType PointType = iota
	StationType
)

type Translatable interface {
	Title(language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error)
	PopularTitle(language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error)
	Preposition(language lang.Lang, preposition lang.Preposition) (string, error)
}

type Point interface {
	Proto() proto.Message
	Type() PointType
	ID() int32
	CountryID() int32
	RegionID() uint32
	SettlementID() int32
	TimeZoneID() int32
	Slug() string
	PointKey() string
	Latitude() float64
	Longitude() float64
	Translatable(translator *i18n.TranslatableFactory) Translatable
}

type Station struct {
	value *rasp.TStation
}

func NewStation(value *rasp.TStation) Point {
	if value == nil {
		return nil
	}
	return &Station{value}
}

func (p *Station) Proto() proto.Message {
	return p.value
}

func (p *Station) Type() PointType {
	return StationType
}

func (p *Station) ID() int32 {
	return p.value.Id
}

func (p *Station) CountryID() int32 {
	return int32(p.value.CountryId)
}

func (p *Station) RegionID() uint32 {
	return p.value.RegionId
}

func (p *Station) SettlementID() int32 {
	return p.value.SettlementId
}

func (p *Station) TimeZoneID() int32 {
	return p.value.TimeZoneId
}

func (p *Station) Slug() string {
	return p.value.Slug
}

func (p *Station) PointKey() string {
	return "s" + strconv.Itoa(int(p.ID()))
}

func (p *Station) Latitude() float64 {
	return p.value.Latitude
}

func (p *Station) Longitude() float64 {
	return p.value.Longitude
}

func (p *Station) Translatable(factory *i18n.TranslatableFactory) Translatable {
	return factory.TranslatableStation(p.value)
}

type Settlement struct {
	Value *rasp.TSettlement
}

func NewSettlement(value *rasp.TSettlement) *Settlement {
	if value == nil {
		return nil
	}
	return &Settlement{value}
}

func (p *Settlement) Proto() proto.Message {
	return p.Value
}

func (p *Settlement) Type() PointType {
	return SettlementType
}

func (p *Settlement) ID() int32 {
	return p.Value.Id
}

func (p *Settlement) CountryID() int32 {
	return p.Value.CountryId
}

func (p *Settlement) RegionID() uint32 {
	return uint32(p.Value.RegionId)
}

func (p *Settlement) SettlementID() int32 {
	return p.ID()
}

func (p *Settlement) TimeZoneID() int32 {
	return p.Value.TimeZoneId
}

func (p *Settlement) Slug() string {
	return p.Value.Slug
}

func (p *Settlement) PointKey() string {
	return "c" + strconv.Itoa(int(p.ID()))
}

func (p *Settlement) Latitude() float64 {
	return p.Value.Latitude
}

func (p *Settlement) Longitude() float64 {
	return p.Value.Longitude
}

func (p *Settlement) Translatable(factory *i18n.TranslatableFactory) Translatable {
	return factory.TranslatableSettlement(p.Value)
}
