package regioncapital

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/geobase"
	dicts "a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/geo"
)

type GeobaseClient interface {
	GetRegionByID(id geobase.ID, crimea ...geobase.CrimeaStatus) (*geobase.Region, error)
}

type Repository struct {
	logger        log.Structured
	geobaseClient GeobaseClient
	repoProvider  *registry.RepositoryRegistry
}

func NewRepository(logger log.Logger, geobaseClient GeobaseClient, repoProvider *registry.RepositoryRegistry) *Repository {
	logger = log.With(logger, log.String("source", "regioncapital.Repository"))
	return &Repository{
		logger:        logger.Structured(),
		geobaseClient: geobaseClient,
		repoProvider:  repoProvider,
	}
}

func (r *Repository) GetRegionCapital(geoID int) *dicts.TSettlement {
	geoObject := r.getGeoObjectByID(geobase.ID(geoID))
	if geoObject == nil {
		return nil
	}

	if settlement := r.getCapitalForRegionGeoObject(geoObject); settlement != nil {
		return settlement
	}

	if !geo.IsSettlementTypes(geoObject.Type) {
		r.logger.Debug(
			"geo object has wrong type",
			log.Int("geo_type", int(geoObject.Type)),
			log.Int("geo_id", geoID),
		)
		return nil
	}

	if settlement := r.getParentSettlement(geoObject); settlement != nil {
		return settlement
	}

	regionGeoObject := r.getParentByType(geoObject, geobase.RegionTypeRegion)
	if regionGeoObject == nil {
		r.logger.Debug(
			"geo object has not region parent",
			log.Int("geo_id", geoID),
		)
		return nil
	}

	repo := r.repoProvider.GetSettlementRepo()

	logFields := []log.Field{
		log.Int("geo_id", geoID),
		log.Int("region_id", int(geoObject.ID)),
	}
	if settlement, found := repo.GetCapitalByRegionID(int32(regionGeoObject.ID)); found {
		r.logger.Info("found region capital in cache", logFields...)
		return settlement
	}

	r.logger.Debug("not found region capital in cache", logFields...)
	return nil
}

func (r *Repository) getCapitalForRegionGeoObject(geoObject *geobase.Region) *dicts.TSettlement {
	repo := r.repoProvider.GetSettlementRepo()
	if settlement, found := repo.GetCapitalByRegionID(int32(geoObject.ID)); found {
		return settlement
	}
	return nil
}

func (r *Repository) getParentSettlement(geoObject *geobase.Region) *dicts.TSettlement {
	// some region -> some fake region capital -> some real city
	geoObject = r.getParentByType(geoObject, geobase.RegionTypeCity)
	if geoObject == nil {
		return nil
	}

	repo := r.repoProvider.GetSettlementRepo()
	if settlement, found := repo.GetByGeoID(int32(geoObject.ID)); found {
		return settlement
	}
	return nil
}

func (r *Repository) getParentByType(geoObject *geobase.Region, targetType geobase.RegionType) *geobase.Region {
	if geoObject == nil {
		return nil
	}
	for {
		geoObject = r.getGeoObjectByID(geoObject.ParentID)
		if geoObject == nil || geoObject.ID == 0 {
			return nil
		}
		if geoObject.Type == targetType {
			return geoObject
		}
	}
}

func (r *Repository) getGeoObjectByID(id geobase.ID) *geobase.Region {
	geoObject, err := r.geobaseClient.GetRegionByID(id)
	if geoObject != nil {
		return geoObject
	}

	fields := []log.Field{log.Int("geo_id", int(id))}
	if err != nil {
		fields = append(fields, log.Error(err))
	}
	r.logger.Debug("geo object not found in geobase", fields...)
	return nil
}
