package schedule

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/limitcondition"
)

const dayMinutes = 24 * 60

var stationsReturnsManySegments = map[int32]bool{
	9607193: true, // Томск
	9607190: true, // Томск
}

func excludeThroughTrains(segments rawSegments) rawSegments {
	type trainKey struct {
		departureStationID  int32
		arrivalStationID    int32
		departureDayMinutes int64
		arrivalDayMinutes   int64
	}

	type trainsGroup map[trainKey]rawSegments
	basicTrains := make(trainsGroup)
	throughTrains := make(trainsGroup)

	for _, segment := range segments {
		startMinutes := int64(segment.thread.TzStartTime) / 60
		key := trainKey{
			segment.departure.StationId,
			segment.arrival.StationId,
			(startMinutes + segment.departure.DepartureTz) % dayMinutes,
			(startMinutes + segment.arrival.ArrivalTz) % dayMinutes,
		}

		if segment.thread.Type == rasp.TThread_TYPE_THROUGH_TRAIN {
			if _, found := basicTrains[key]; found {
				continue
			}
			throughTrains[key] = append(throughTrains[key], segment)
		} else {
			delete(throughTrains, key)
			basicTrains[key] = append(basicTrains[key], segment)
		}
	}

	var result rawSegments
	for _, group := range []trainsGroup{basicTrains, throughTrains} {
		for _, filteredSegments := range group {
			result = append(result, filteredSegments...)
		}
	}
	return result
}

func filterSegmentsByMajority(logger log.Structured, segments rawSegments, condition *limitcondition.LimitCondition) rawSegments {
	var newSegments rawSegments
	for _, rawSegment := range segments {
		if !rawSegment.departure.IsSearchableFrom || rawSegment.departureMajorityID > int32(condition.DepartureMaxMajorityID) {
			logger.Debug("exclude rawSegment by departure majority")
			continue
		}

		if !rawSegment.arrival.IsSearchableTo || rawSegment.arrivalMajorityID > int32(condition.ArrivalMaxMajorityID) {
			logger.Debug("exclude rawSegment by arrival majority")
			continue
		}

		if rawSegment.departure.IsDepartureCodeSharing && rawSegment.arrival.IsArrivalCodeSharing {
			logger.Debug("exclude rawSegment by code sharing")
			continue
		}

		newSegments = append(newSegments, rawSegment)
	}
	return newSegments
}

// Если несколько станций отправления или назначения входят в STATIONS_RETURNS_MANY_SEGMENTS,
// то будет возвращен _find_best_segment для каждой группировки (departure.station_id, arrival.station_id).
// При этом пары (departure, arrival), не входящие в STATIONS_RETURNS_MANY_SEGMENTS будут входить в одну из созданных
// групп, просто чтобы не потерялись, вдруг они лучше всех.
func findBestSegments(segments rawSegments) (result rawSegments) {
	type routeKey struct {
		departureStationID, arrivalStationID int32
	}
	segmentGroups := make(map[routeKey]rawSegments)

	for _, segment := range segments {
		key := routeKey{}
		if stationsReturnsManySegments[segment.departure.StationId] {
			key.departureStationID = segment.departure.StationId
		}
		if stationsReturnsManySegments[segment.arrival.StationId] {
			key.arrivalStationID = segment.arrival.StationId
		}
		segmentGroups[key] = append(segmentGroups[key], segment)
	}

	defaultKey := routeKey{}
	if len(segmentGroups) > 1 && len(segmentGroups[defaultKey]) != 0 {
		defaultGroup := segmentGroups[defaultKey]
		delete(segmentGroups, defaultKey)

		for key := range segmentGroups {
			segmentGroups[key] = append(segmentGroups[key], defaultGroup...)
			break
		}
	}

	for _, group := range segmentGroups {
		if best := findBestSegment(group); best != nil {
			result = append(result, best)
		}
	}
	return
}

// Получаем первым элементом рейс от станции с максимальным приоритетом до станции с максимальным приоритетом,
// если станций с максимальным приоритетом несколько, то от самой поздней до самой ранней из них.
func findBestSegment(segments rawSegments) *rawSegment {
	if len(segments) == 0 {
		return nil
	}

	less := func(lhs, rhs *rawSegment) bool {
		if lhs.departureMajorityID != rhs.departureMajorityID {
			return lhs.departureMajorityID < rhs.departureMajorityID
		}
		if lhs.arrivalMajorityID != rhs.arrivalMajorityID {
			return lhs.arrivalMajorityID < rhs.arrivalMajorityID
		}

		// начальная станция отправления более приоритетная
		if lhs.departure.HasArrival != rhs.departure.HasArrival {
			return !lhs.departure.HasArrival
		}
		if lhs.departure.DepartureTz != rhs.departure.DepartureTz {
			return lhs.departure.DepartureTz > rhs.departure.DepartureTz
		}
		return lhs.arrival.ArrivalTz < rhs.arrival.ArrivalTz
	}

	bestSegment := segments[0]
	for i := 1; i < len(segments); i++ {
		s := segments[i]
		if less(s, bestSegment) {
			bestSegment = s
		}
	}
	return bestSegment
}
