package schedule

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/consts"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date/daytime"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date/runmask"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
)

const searchDaysLimit = 100

type SegmentProvider struct {
	logger   log.Logger
	registry *registry.RepositoryRegistry
}

func (m *SegmentProvider) SegmentsByThreadRuns(segment *rawSegment, minDepartureDate time.Time) Segments {
	minDepartureDate = m.getLocalDepartureDate(segment.departure, minDepartureDate)
	var result Segments
	for _, threadStart := range m.getThreadRuns(segment.thread, minDepartureDate, searchDaysLimit) {
		departure := segment.departure
		arrival := segment.arrival

		departureDateTime := m.getStationEventDateTime(threadStart, departure, departure.DepartureTz)
		result = append(result, &Segment{
			Thread:            segment.thread,
			ThreadStartDt:     threadStart,
			Departure:         departure,
			DepartureDateTime: departureDateTime,
			Arrival:           segment.arrival,
			ArrivalDateTime:   m.getStationEventDateTime(threadStart, arrival, arrival.ArrivalTz),
			EventTime:         departureDateTime,
			EventLocation:     m.getEventLocation(departure.StationId),
		})
	}
	return result
}

func (m *SegmentProvider) getStationEventDateTime(threadStart time.Time, station *rasp.TThreadStation, stationTz int64) time.Time {
	stationEventDateTime := threadStart.Add(time.Duration(stationTz) * time.Minute)
	timeZoneRepo := m.registry.GetTimeZoneRepo()
	if location, found := timeZoneRepo.GetLocationByID(station.TimeZoneId); found {
		return date.ReplaceLocation(stationEventDateTime, location)
	}
	return stationEventDateTime
}

func (m *SegmentProvider) getLocalDepartureDate(departure *rasp.TThreadStation, minDepartureDate time.Time) time.Time {
	minDepartureDate = m.localize(minDepartureDate, departure.TimeZoneId)
	if departure.HasDeparture {
		minDepartureDate = minDepartureDate.Add(-time.Duration(departure.DepartureTz) * time.Minute)
	}
	return minDepartureDate
}

func (m *SegmentProvider) getThreadRuns(thread *rasp.TThread, minDepartureDate time.Time, searchDays int) []time.Time {
	mask := runmask.New(thread.YearDays)
	threadStart := daytime.DayTime(thread.TzStartTime).Combine(minDepartureDate)
	if threadStart.Before(minDepartureDate) {
		threadStart = threadStart.Add(units.Day)
	}

	var result []time.Time
	for ; searchDays > 0; searchDays-- {
		if mask.RunsAt(threadStart) {
			result = append(result, threadStart)
		}
		threadStart = threadStart.Add(units.Day)
	}
	return result
}

func (m *SegmentProvider) localize(date time.Time, timeZoneID int32) time.Time {
	timeZoneRepo := m.registry.GetTimeZoneRepo()
	if location, found := timeZoneRepo.GetLocationByID(timeZoneID); found {
		return date.In(location)
	}
	return date
}

func (m *SegmentProvider) getEventLocation(stationID int32) *time.Location {
	stationRepo := m.registry.GetStationRepo()
	station, found := stationRepo.Get(stationID)
	if !found {
		return consts.DefaultLocation
	}

	timeZoneRepo := m.registry.GetTimeZoneRepo()
	if location, found := timeZoneRepo.GetLocationByID(station.TimeZoneId); found {
		return location
	}
	return consts.DefaultLocation
}
