package templater

import (
	"fmt"

	"github.com/flosch/pongo2/v4"

	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
)

var (
	onlySupportedLanguage   = lang.Ru
	onlySupportedTranslator *i18n.TranslatableFactory
)

func SetTranslator(t *i18n.TranslatableFactory) {
	onlySupportedTranslator = t
}

func joinLastFilter(in *pongo2.Value, param *pongo2.Value) (out *pongo2.Value, err *pongo2.Error) {
	separator := ""
	if !in.CanSlice() || in.Len() < 2 {
		return in, nil
	}
	if param.IsString() {
		separator = param.String()
	}
	result := make([]interface{}, in.Len()-1)
	for i := 0; i < in.Len()-2; i++ {
		result[i] = in.Index(i).Interface()
	}
	result[in.Len()-2] = in.Index(in.Len()-2).String() + separator + in.Index(in.Len()-1).String()
	return pongo2.AsValue(result), nil
}

func translatable(in *pongo2.Value) (points.Translatable, error) {
	translatable, ok := in.Interface().(points.Translatable)
	if !ok {
		point, ok := in.Interface().(points.Point)
		if !ok {
			return nil, fmt.Errorf("value is not supported")
		}
		translatable = point.Translatable(onlySupportedTranslator)
	}
	return translatable, nil
}

func newPongoErrorOrNil(sender string, err error) *pongo2.Error {
	if err == nil {
		return nil
	}
	return &pongo2.Error{
		Sender:    sender,
		OrigError: err,
	}
}

func translatePoint(in *pongo2.Value, gCase lang.GrammaticalCase) (string, error) {
	translatable, err := translatable(in)
	if err != nil {
		return "", err
	}
	title, err := translatable.Title(onlySupportedLanguage, gCase)
	if err != nil {
		return "", err
	}
	return title, nil
}

// имя города в именительном падеже, например Москва
func nominativeFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	title, err := translatePoint(in, lang.Nominative)
	return pongo2.AsValue(title), newPongoErrorOrNil("filter:nominative", err)
}

// имя города в родительном падеже, например Москвы
func genitiveFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	title, err := translatePoint(in, lang.Genitive)
	return pongo2.AsValue(title), newPongoErrorOrNil("filter:genitive", err)
}

// имя города в дательном падеже, например Москве
func dativeFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	title, err := translatePoint(in, lang.Dative)
	return pongo2.AsValue(title), newPongoErrorOrNil("filter:dative", err)
}

// имя города в винительном падеже, например Москву
func accusativeFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	title, err := translatePoint(in, lang.Accusative)
	return pongo2.AsValue(title), newPongoErrorOrNil("filter:accusative", err)
}

// имя города в предложном падеже, например Москве
func prepositionalFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	title, err := translatePoint(in, lang.Locative)
	return pongo2.AsValue(title), newPongoErrorOrNil("filter:prepositional", err)
}

// предлог (в, на) для предложного падежа
func prepositionInFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	filterName := "filter:preposition_in"
	translatable, err := translatable(in)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	pp, err := translatable.Preposition(onlySupportedLanguage, lang.PrepositionIn)
	return pongo2.AsValue(pp), newPongoErrorOrNil(filterName, err)
}

func prepositionToFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	filterName := "filter:preposition_to"
	translatable, err := translatable(in)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	pp, err := translatable.Preposition(onlySupportedLanguage, lang.PrepositionTo)
	return pongo2.AsValue(pp), newPongoErrorOrNil(filterName, err)
}

// тоже самое что и "{{city.preposition}} {{city.prepositional}}", например "в Москве"
func locativeFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	filterName := "filter:locative"
	t, err := translatable(in)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	preposition, err := t.Preposition(onlySupportedLanguage, lang.PrepositionIn)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	title, err := t.Title(onlySupportedLanguage, lang.Locative)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	result := title
	if preposition != "" {
		result = preposition + " " + title
	}
	return pongo2.AsValue(result), nil
}

// тоже самое что и "{{city.preposition}} {{city.accusative}}", например "в Москву"
func directionalFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	filterName := "filter:directional"
	t, err := translatable(in)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	preposition, err := t.Preposition(onlySupportedLanguage, lang.PrepositionTo)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	title, err := t.Title(onlySupportedLanguage, lang.Accusative)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	result := title
	if preposition != "" {
		result = preposition + " " + title
	}
	return pongo2.AsValue(result), nil
}

// тоже самое что и "из {{city.genitive}}", например "из Москвы"
func ablativeFilter(in *pongo2.Value, param *pongo2.Value) (*pongo2.Value, *pongo2.Error) {
	filterName := "filter:ablative"
	t, err := translatable(in)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	preposition, err := t.Preposition(onlySupportedLanguage, lang.PrepositionFrom)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	title, err := t.Title(onlySupportedLanguage, lang.Genitive)
	if err != nil {
		return nil, newPongoErrorOrNil(filterName, err)
	}
	result := title
	if preposition != "" {
		result = preposition + " " + title
	}
	return pongo2.AsValue(result), nil
}
