package threadtitle

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
)

const dash = "\u2014"

type Generator struct {
	logger               log.Structured
	repoRegistry         *registry.RepositoryRegistry
	translatableFactory  *i18n.TranslatableFactory
	trainTitleTranslator *i18n.TrainTitleTranslator
}

func NewGenerator(logger log.Logger, repoRegistry *registry.RepositoryRegistry, translatedFactory *i18n.TranslatableFactory, trainTitleTranslator *i18n.TrainTitleTranslator) *Generator {
	return &Generator{
		logger:               logger.Structured(),
		repoRegistry:         repoRegistry,
		translatableFactory:  translatedFactory,
		trainTitleTranslator: trainTitleTranslator,
	}
}

func (g *Generator) FormatThreadTitle(language lang.Lang, threadTitle *rasp.TThreadTitle) (string, error) {
	titles, err := g.makePartTitles(language, threadTitle.TitleParts)
	if err != nil {
		return "", generatorFormatThreadTitleFn.WrapError(err)
	}
	routeTitle := strings.Join(titles, fmt.Sprintf(" %s ", dash))

	if threadTitle.IsRing {
		routeTitle, err = g.trainTitleTranslator.FormatRouteTitleAsRing(language, routeTitle)
		if err != nil {
			return "", generatorFormatThreadTitleFn.WrapError(err)
		}
	}

	if threadTitle.IsCombined {
		if threadTitle.Type == rasp.TThreadTitle_TYPE_SUBURBAN {
			routeTitle, err = g.trainTitleTranslator.FormatRouteTitleAsConfirmedTrain(language, routeTitle)
			if err != nil {
				return "", generatorFormatThreadTitleFn.WrapError(err)
			}
		} else if threadTitle.Type == rasp.TThreadTitle_TYPE_DEFAULT {
			if len(threadTitle.TitleParts) == 3 {
				routeTitle, err = g.trainTitleTranslator.FormatRouteTitleWithTransfer(language, routeTitle)
				if err != nil {
					return "", generatorFormatThreadTitleFn.WrapError(err)
				}
			} else {
				routeTitle, err = g.trainTitleTranslator.FormatRouteTitleWithManyTransfers(language, routeTitle)
				if err != nil {
					return "", generatorFormatThreadTitleFn.WrapError(err)
				}
			}
		}
	}
	return routeTitle, nil
}

func (g *Generator) makePartTitles(language lang.Lang, parts []*rasp.TThreadTitlePart) ([]string, error) {
	var titles []string
	for _, part := range parts {
		if part.StationId != 0 {
			station, found := g.repoRegistry.GetStationRepo().Get(int32(part.StationId))
			if !found {
				return nil, fmt.Errorf("%s: station not found by id=%d", generatorMakePartTitlesFn, part.StationId)
			}

			title, err := g.translatableFactory.TranslatableStation(station).Title(language, lang.Nominative)
			if err != nil {
				g.logger.Warn("translated title not found",
					log.UInt32("station_id", part.StationId),
					log.Error(err),
				)
			}
			titles = append(titles, title)
		} else if part.SettlementId != 0 {
			settlement, found := g.repoRegistry.GetSettlementRepo().Get(int32(part.SettlementId))
			if !found {
				return nil, fmt.Errorf("%s: settlement not found by id=%d", generatorMakePartTitlesFn, part.SettlementId)
			}

			title, err := g.translatableFactory.TranslatableSettlement(settlement).Title(language, lang.Nominative)
			if err != nil {
				g.logger.Warn("translated title not found",
					log.UInt32("settlement_id", part.SettlementId),
					log.Error(err),
				)
			}
			titles = append(titles, title)
		} else {
			return nil, fmt.Errorf("%s: empty title parts", generatorMakePartTitlesFn)
		}
	}
	return titles, nil
}
