package traincity

import (
	"sort"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
)

type Cache struct {
	logger log.Structured
	repos  *registry.RepositoryRegistry

	settlementIDToStation map[int32]*rasp.TStation
}

func NewCache(logger log.Structured, registry *registry.RepositoryRegistry) *Cache {
	return &Cache{
		logger: logger,
		repos:  registry,

		settlementIDToStation: make(map[int32]*rasp.TStation),
	}
}

type byMajority []*rasp.TStation

func (t byMajority) Len() int           { return len(t) }
func (t byMajority) Swap(i, j int)      { t[i], t[j] = t[j], t[i] }
func (t byMajority) Less(i, j int) bool { return t[i].Majority < t[j].Majority }

func (c *Cache) Build() *Cache {
	var fakeStations, stations []*rasp.TStation

	stationRepo := c.repos.GetStationRepo()
	for _, code := range c.repos.GetStationCodeRepo().GetCodesBySystemID(rasp.ECodeSystem_CODE_SYSTEM_EXPRESS) {
		station, found := stationRepo.Get(code.StationId)
		if !found || station.TransportType != rasp.TTransport_TYPE_TRAIN || station.SettlementId == 0 {
			continue
		}

		if station.Majority == rasp.TStation_MAJORITY_EXPRESS_FAKE {
			fakeStations = append(fakeStations, station)
		}

		if !station.IsHidden {
			stations = append(stations, station)
		}
	}

	sort.Sort(byMajority(stations))
	for _, station := range append(fakeStations, stations...) {
		if _, found := c.settlementIDToStation[station.SettlementId]; !found {
			c.settlementIDToStation[station.SettlementId] = station
		}
	}
	return c
}

func (c *Cache) Get(id int32) *rasp.TStation {
	if station, found := c.settlementIDToStation[id]; found {
		return station
	}
	return nil
}
