package traincity

import (
	"sync"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/updater"
)

type Repository struct {
	logger   log.Structured
	registry *registry.RepositoryRegistry

	updateMutex sync.Mutex
	cacheValue  atomic.Value
	validCache  bool
}

func NewRepository(logger log.Logger, repoRegistry *registry.RepositoryRegistry) *Repository {
	repo := &Repository{
		logger:   logger.Structured(),
		registry: repoRegistry,
	}

	repoRegistry.AddRepositoryObserver(registry.Station, repo)
	repoRegistry.AddRepositoryObserver(registry.StationCode, repo)
	return repo
}

func (r *Repository) OnDictUpdate(_ registry.DictType) {
	r.validCache = false
}

func (r *Repository) UpdateCache() error {
	r.updateMutex.Lock()
	defer r.updateMutex.Unlock()

	if r.validCache {
		return updater.AlreadyUpdated
	}

	cache := NewCache(r.logger, r.registry).Build()
	r.cacheValue.Store(cache)
	r.validCache = true
	return nil
}

func (r *Repository) Get(settlementID int32) *rasp.TStation {
	cache := r.cacheValue.Load().(*Cache)
	return cache.Get(settlementID)
}
